/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import static nz.org.riskscape.engine.Matchers.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.stream.Collectors;
import java.util.stream.LongStream;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.rl.agg.AggregationFunction;
import nz.org.riskscape.engine.rl.agg.BaseAggregationFunctionTest;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.rl.ast.ExpressionProblems;
import nz.org.riskscape.rl.ast.Lambda;
import nz.org.riskscape.rl.ast.PropertyAccess;

public class BucketFunctionTest extends BaseAggregationFunctionTest {

  Struct inputType = Types.INTEGER.asStruct();
  BucketFunction bucketFunction = new BucketFunction();

  @Before
  public void setup() {
    project.getFunctionSet().add(AggregationFunction.asFunction(bucketFunction).builtin("bucket", Category.MISC));
  }

  @Test
  public void canCountOccurrencesOfADistinctValues() {
    inputType = Types.TEXT.asStruct();
    tuples = Arrays.asList("foo", "bar", "baz", "foo", "bar", "foo").stream().map(str -> Tuple.ofValues(inputType, str))
        .collect(Collectors.toList());

    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> bucket = value,"
        + "  select: count(*),"
        + "  buckets: {foos: 'foo', bars: 'bar', bazzes: 'baz', mystery: 'wooooo'}"
        + ")"));

    process();

    assertThat(serialResult, equalTo(tuple("{foos: 3, bars: 2, bazzes: 1, mystery: 0}")));
  }

  @Test
  public void canBucketBasedOnRanges() throws Exception {
    tuples = LongStream.range(0L, 100L).mapToObj(aLong -> Tuple.ofValues(inputType, aLong))
        .collect(Collectors.toList());

    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> value >= bucket.start && value < bucket.end,"
        + "  select: count(*),"
        + "  buckets: {foo: {start: 0, end: 13}, bar: {start: 13, end: 9999999}}"
        + ")"));

    process();

    assertThat(serialResult, equalTo(tuple("{foo: 13, bar: 87}")));
  }

  @Test
  public void isBucketedToAllBucketsThatMatch() throws Exception {
    tuples = LongStream.range(0L, 100L).mapToObj(aLong -> Tuple.ofValues(inputType, aLong))
        .collect(Collectors.toList());

    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> value <= bucket,"
        + "  select: sum(value),"
        + "  buckets: {max_ten: 10, max_50: 50, big: 99999999}"
        + ")"));

    process();

    Struct expectedResultType = Struct.of(
        "max_ten", Nullable.INTEGER,
        "max_50", Nullable.INTEGER,
        "big", Nullable.INTEGER
    );

    assertThat(serialResult, equalTo(Tuple.ofValues(expectedResultType, 55L, 1275L, 4950L)));

  }

//  @Test
  // this fails because the if_null doesn't do anything - to fix this, we'll need to change the accumulator API - I
  // might come back to this to attempt to make it sound
  public void selectStatementCanProvideAnIdentityValue() throws Exception {
    tuples = LongStream.range(0L, 100L).mapToObj(aLong -> Tuple.ofValues(inputType, aLong))
        .collect(Collectors.toList());

    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> value <= bucket,"
        + "  select: if_null(sum(value), 0),"
        + "  buckets: {max_ten: 10, max_50: 50, big: 99999999}"
        + ")"));

    process();

    Struct expectedResultType = Struct.of(
        "max_ten", Types.INTEGER,
        "max_50", Types.INTEGER,
        "big", Types.INTEGER
    );

    assertThat(serialResult, equalTo(Tuple.ofValues(expectedResultType, 55L, 1275L, 4950L)));
  }

  @Test
  public void pickMustBeALambdaExpressionWithOneArgument() throws Exception {
    // not even a lambda
    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: value < 10,"
        + "  select: sum(value),"
        + "  buckets: {max_ten: 10, max_50: 50, big: 99999999}"
        + ")"));

    assertThat(
      realizationProblems,
      contains(hasAncestorProblem(equalTo(ExpressionProblems.get().mismatch(
        parse("value < 10"),
        Lambda.class,
        "bucket_value -> exposure.category = bucket_value"
      ))))
    );

    // wrong number of args
    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: (bucket, value) -> bucket > value,"
        + "  select: sum(value),"
        + "  buckets: {max_ten: 10, max_50: 50, big: 99999999}"
        + ")"));

    assertThat(
      realizationProblems,
      contains(hasAncestorProblem(equalTo(
        ExpressionProblems.get().lambdaArityError(parse("(bucket, value) -> bucket > value"), 2, 1)
      ))
    ));
  }

  @Test
  public void selectMustBeAnAggregateExpression() throws Exception {
    // value not surrounded by aggregate function
    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> value < bucket,"
        + "  select: value,"
        + "  buckets: {max_ten: 10, max_50: 50, big: 99999999}"
        + ")"));

    assertThat(
      realizationProblems,
      contains(hasAncestorProblem(equalTo(
        problemWith(
          "select",
          ExpressionProblems.get().propertyOutsideOfAggregationFunction(parse("value").isA(PropertyAccess.class).get())
        )
      )))
    );
  }

  @Test
  public void bucketsMustBeAConstantStruct() throws Exception {
    // array, not a struct
    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> true,"
        + "  select: count(*),"
        + "  buckets: [0, 10, 100, 100]"
        + ")"));

    assertThat(
        realizationProblems,
        contains(hasAncestorProblem(equalTo(
          problemWith(
            "buckets",
            TypeProblems.get().mismatch(parse("[0, 10, 100, 100]"), Struct.EMPTY_STRUCT, RSList.create(Types.INTEGER))
          )
        )))
      );

    // buckets not constant
    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> true,"
        + "  select: count(*),"
        + "  buckets: {max_ten: value, max_50: 50, big: 99999999}"
        + ")"));

    assertThat(
        realizationProblems,
        contains(hasAncestorProblem(equalTo(
          problemWith(
            "buckets",
            ExpressionProblems.get().constantRequired(parse("{max_ten: value, max_50: 50, big: 99999999}"))
          )
        )))
      );
  }

  // a lot of these tests are actually asserting limitations of the behaviour - we could realize an expression for each
  // bucket, and as long as they all can yield a boolean then sure
  @Test
  public void bucketsMustBeOfTheExactSameType() throws Exception {
    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> true,"
        + "  select: count(*),"
        + "  buckets: {max_ten: 1, max_50: 50.0}"
        + ")"));

    // types can't change, even where they might all realize
    assertThat(
      realizationProblems,
      contains(hasAncestorProblem(equalTo(
        TypeProblems.get().mismatch("bucket max_50", Types.INTEGER, Types.FLOATING)
      )))
    );

    // nullability can't change either
    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> true,"
        + "  select: count(*),"
        + "  buckets: {max_ten: 1, max_50: null_of('integer')}"
        + ")"));

    assertThat(
      realizationProblems,
      contains(hasAncestorProblem(equalTo(
        TypeProblems.get().mismatch("bucket max_50", Types.INTEGER, Nullable.INTEGER)
      )))
    );

    // structs must be identical, too, even if we're not using all the members
    realizeAggregate(inputType, parse(""
        + "bucket("
        + "  pick: bucket -> bucket.foo > value,"
        + "  select: count(*),"
        + "  buckets: {a: {foo: 1}, b: {foo: 2}, c: {foo: 3, bar: 4}}"
        + ")"));

    assertThat(
      realizationProblems,
      contains(hasAncestorProblem(equalTo(
        TypeProblems.get().mismatch(
            "bucket c",
            Struct.of("foo", Types.INTEGER),
            Struct.of("foo", Types.INTEGER, "bar", Types.INTEGER)
        )
      )))
    );
  }

  private Problem problemWith(String argsName, Problem... children) {
    return Problems.foundWith(bucketFunction.getArguments().get(argsName), children);
  }
}
