/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.data;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.nio.file.Path;

import org.geotools.referencing.CRS;
import org.geotools.referencing.crs.DefaultEngineeringCRS;
import org.junit.Test;
import org.geotools.api.referencing.FactoryException;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.data.Bookmark;
import nz.org.riskscape.engine.defaults.data.ShapefileBookmarkResolver;
import nz.org.riskscape.engine.geo.PrjParser;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.relation.SpatialMetadata;

public class ShapefileCrsTest extends ProjectTest {

  Path shapefilePath;
  ShapefileBookmarkResolver resolver = new ShapefileBookmarkResolver(engine);
  Relation relation;
  CoordinateReferenceSystem crs;

  @Test
  public void testAStandardShapefile() throws Exception {
    open("shapefile_wgs84.shp");
    assertTrue(CRS.equalsIgnoreMetadata(CRS.decode("EPSG:4326", true), crs));
    assertThat(relation.size().get(), is(2L));
  }

  @Test
  public void shapefileWithNoPrjFile() throws Exception {
    open("shapefile_wgs852.shp");
    shapefilePath = getPackagePath().resolve("shapefile_wgs852.shp");
    assertEquals(DefaultEngineeringCRS.GENERIC_2D, crs);
  }

  @Test
  public void shapefileWithWeirdPrj() throws Exception {
    engine.addFeature((PrjParser) prjString -> {
      try {
        return CRS.decode(prjString.trim());
      } catch (FactoryException e) {
        throw new RuntimeException(e);
      }
    });

    open("shapefile_wgs85.shp");
    assertEquals(CRS.decode("EPSG:4167"), crs);
  }

  private void open(String name) {
    shapefilePath = getPackagePath().resolve(name);
    relation =
        resolver.resolve(Bookmark.fromURI(shapefilePath.toUri()), bindingContext).get().getData(Relation.class).get();

    crs = relation.getSpatialMetadata().map(SpatialMetadata::getCrs).orElse(null);
  }

}
