/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.curves;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.apache.commons.math3.analysis.UnivariateFunction;
import org.apache.commons.math3.analysis.polynomials.PolynomialFunction;
import org.junit.Before;
import org.junit.Test;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.function.UnaryFunction;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.engine.types.FunctionType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

public class TrapezoidIntegrationFunctionTest extends BaseExpressionRealizerTest {

  @Before
  public void before() {
    project.getFunctionSet().add(new TrapezoidIntegrationFunction().identified("trapz"));
  }

  @Test
  public void testIntegratingASimpleLinearFunction() {
    // nice, simple linear function that's a straight line across the x-axis
    RiskscapeFunction function = unaryFunctionOn(new PolynomialFunction(new double[] {1, 0}));

    Struct inputType = FunctionType.fromFunction(function).asStruct();

    evaluate("trapz(value, 1.0, 3.0, 2)", Tuple.ofValues(inputType, function));
    assertEquals(2D, (Double) evaluated, 0.001);

    evaluate("trapz(value, 1.0, 3.0, 2000)", Tuple.ofValues(inputType, function));
    assertEquals(2D, (Double) evaluated, 0.001);

    evaluate("trapz(value, -1.0, 5.0, 2000)", Tuple.ofValues(inputType, function));
    assertEquals(6D, (Double) evaluated, 0.001);
  }

  @Test
  public void testIntegratingAPolynomialFunction() {
    RiskscapeFunction function = unaryFunctionOn(new PolynomialFunction(new double[] {1, 1, 1}));

    Struct inputType = FunctionType.fromFunction(function).asStruct();

    evaluate("trapz(value, -1.0, 1.0, 10)", Tuple.ofValues(inputType, function));
    assertEquals(2.68D, (Double) evaluated, 0.001);

  }

  @Test
  public void testIntegratingA45DegreeSlope() {
    RiskscapeFunction function = RiskscapeFunction.create("test", Lists.newArrayList(Types.FLOATING),
        Types.FLOATING, (x) -> x.get(0));
    Struct inputType = FunctionType.fromFunction(function).asStruct();

    evaluate("trapz(value, 8.0, 10.0, 2)", Tuple.ofValues(inputType, function));
    assertEquals(18D, (Double) evaluated, 0.001);

    // what happens if we mess up the start point order with high -> low
    evaluate("trapz(value, 10.0, 8.0, 2)", Tuple.ofValues(inputType, function));
    assertEquals(18D, (Double) evaluated, 0.001);
  }

  @Test
  public void failsToRealizeFunctionBadReturnType() {
    RiskscapeFunction function = RiskscapeFunction.create("test", Lists.newArrayList(Types.FLOATING),
        Types.TEXT, (x) -> "foo");
    Struct inputType = FunctionType.fromFunction(function).asStruct();

    assertThat(realizeOnly("trapz(value, 10.0, 8.0, 2)", inputType), Matchers.failedResult(
        Matchers.hasAncestorProblem(is(TrapezoidIntegrationFunction.PROBLEMS.functionBadReturnType(Types.TEXT)))
    ));
  }

  @Test
  public void failsToRealizeFunctionRequiresTooManyArgs() {
    RiskscapeFunction function = RiskscapeFunction.create("test", Lists.newArrayList(Types.FLOATING, Types.FLOATING),
        Types.FLOATING, (x) -> x.get(0));
    Struct inputType = FunctionType.fromFunction(function).asStruct();

    assertThat(realizeOnly("trapz(value, 10.0, 8.0, 2)", inputType), Matchers.failedResult(
        Matchers.hasAncestorProblem(is(TrapezoidIntegrationFunction.PROBLEMS.functionBadArgumentTypes(
            Lists.newArrayList(Types.FLOATING, Types.FLOATING))))
    ));
  }

  @Test
  public void failsToRealizeFunctionDoesNotAcceptFloating() {
    RiskscapeFunction function = RiskscapeFunction.create("test", Lists.newArrayList(Types.TEXT),
        Types.FLOATING, (x) -> 10.0D);
    Struct inputType = FunctionType.fromFunction(function).asStruct();

    assertThat(realizeOnly("trapz(value, 10.0, 8.0, 2)", inputType), Matchers.failedResult(
        Matchers.hasAncestorProblem(is(TrapezoidIntegrationFunction.PROBLEMS.functionBadArgumentTypes(
            Lists.newArrayList(Types.TEXT))))
    ));
  }

  RiskscapeFunction unaryFunctionOn(UnivariateFunction wrapping) {
    return new UnaryFunction<Double>(Types.FLOATING, Types.FLOATING, aDouble -> wrapping.value(aDouble));
  }

}
