/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.curves;

import static nz.org.riskscape.engine.Assert.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import com.google.common.collect.Lists;

import lombok.Getter;

public class ContinuousLinearFitterTest extends BaseCurveFitterTest {

  @Getter
  ContinuousLinearFitter fitter = new ContinuousLinearFitter();

  @Test
  public void testLinearFit() {
    xValues.addAll(Lists.newArrayList(10D, 20D, 30D, 40D));
    yValues.addAll(Lists.newArrayList(10D, 10D, 20D, -20D));

    fit();

    // Sanity check that if you check the x values that the curve was built from, you get the same y values
    assertThat(interpolate(10D), is(10D));
    assertThat(interpolate(20D), is(10D));
    assertThat(interpolate(30D), is(20D));
    assertThat(interpolate(40D), is(-20D));

    // smaller x values than curve was built with
    assertThat(interpolate(9D), is(10D));
    assertThat(interpolate(0D), is(10D));
    assertThat(interpolate(-1000D), is(10D));

    // larger x values than curve was built with
    assertThat(interpolate(41D), is(-20D));
    assertThat(interpolate(100D), is(-20D));
    assertThat(interpolate(1000D), is(-20D));

    // midpoints
    assertThat(interpolate(15D), is(10D));
    assertThat(interpolate(25D), is(15D));
    assertThat(interpolate(35D), is(0D));

    // the default score is zero, because otherwise it would be too perfect.
    assertThat(curve.getFitScore(), is(0D));
  }

  @Test
  public void usesTheGivenScore() {
    xValues.addAll(Lists.newArrayList(10D, 20D, 30D, 40D));
    yValues.addAll(Lists.newArrayList(10D, 10D, 20D, -20D));
    addParam("score", 89D);

    fit();

    // the default score is zero, because otherwise it would be too perfect.
    assertThat(curve.getFitScore(), is(89D));
  }

  @Test
  public void cannotHaveDuplicateXValues() {
    xValues.addAll(Lists.newArrayList(10D, 30D, 30D, 40D));
    yValues.addAll(Lists.newArrayList(10D, 15D, 20D, -20D));

    CannotFitCurveException ex = assertThrows(CannotFitCurveException.class, () -> fit());
    assertThat(ex.getProblem(), is(ContinuousLinearFitter.PROBLEMS.duplicateXValues("continuous", 30D)));
  }

  @Test
  public void canHandleNullOrEmptyData() {
    // don't populate xValues/yValues at all, i.e. ObservedPoints is empty
    CannotFitCurveException ex = assertThrows(CannotFitCurveException.class, () -> fit());
    assertThat(ex.getProblem(), is(ContinuousLinearFitter.PROBLEMS.noData()));
  }
}
