/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import java.util.Arrays;
import java.util.stream.Collectors;

import org.junit.Before;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinRange;
import nz.org.riskscape.engine.typexp.DefaultTypeBuilder;
import nz.org.riskscape.engine.typexp.TypeBuilder;

public class ReturnTypeInfererTest extends ProjectTest {

  BindingContext context = super.bindingContext;
  ExpressionRealizer realizer = super.expressionRealizer;
  ClassifierFunctionParser parser = new ClassifierFunctionParser();
  TypeBuilder builder = new DefaultTypeBuilder(project.getTypeSet());

  AST.FunctionDecl parsed;
  RealizedTreeFilter filter;
  ReturnTypeInferer adaptor;

  @Before
  public void before() {
    project.getFunctionSet().insertLast(new DefaultOperators());
  }

  public void parse(String... lines) {
    parsed = parser.parse(Arrays.asList(lines).stream().collect(Collectors.joining("\n")) + "\n");

    parsed.parseTypes(builder);
    parsed.parseExpressions(expressionParser);
  }

  public void build(String... lines) {
    parse(lines);
    filter = RealizedTreeFilter.build(realizer, parsed, parsed.getArgumentTypesDecl().built).orElse(null);
    adaptor = ReturnTypeInferer.build(filter, project.getTypeSet()).get();
  }

  @Test
  public void givesNullIfNoFiltersMatch() throws Exception {
    build(
        "argument-types:",
        "  celsius: integer",
        "filter: celsius > 50",
        "  function: 'cool'"
    );
    assertFalse(filter.isDefaultPresent());
    assertEquals(Nullable.TEXT, adaptor.getResultType());
    assertEquals("cool", adaptor.evaluate(tuple(51L)));
    assertNull(adaptor.evaluate(tuple(49L)));
  }

  @Test
  public void resultTypeHasNullableAttributes() throws Exception {
    build(
        "argument-types:",
        "  celsuis: integer",
        "filter: celsuis > 50",
        "  function:",
        "    a: 'cool'",
        "    b: 'foo'",
        "default:",
        "  a: 'unknown'"
    );

    //The inferred type has the nullable on b as expected.
    Struct expectedType = Struct.of("a", Types.TEXT, "b", Nullable.TEXT);
    assertThat(adaptor.getResultType(), is(expectedType));

    // but the actual results don't, as they are passing the tuple back that was returned by the
    // tree expression.
    assertThat(adaptor.evaluate(tuple(52L)), is(
        Tuple.ofValues(expectedType, "cool", "foo")));

    assertThat(adaptor.evaluate(tuple(30L)), is(
        Tuple.ofValues(expectedType, "unknown")));
  }

  @Test
  public void resultTypeHasNullableStructWhenNoDefault() throws Exception {
    build(
        "argument-types:",
        "  celsuis: integer",
        "filter: celsuis > 50",
        "  function:",
        "    a: 'cool'",
        "    b: 'foo'"
    );

    Struct expectedType = Struct.of("a", Types.TEXT, "b", Types.TEXT);
    assertThat(adaptor.getResultType(), is(Nullable.of(expectedType)));

    assertThat(adaptor.evaluate(tuple(52L)), is(
        Tuple.ofValues(expectedType, "cool", "foo")));

    assertThat(adaptor.evaluate(tuple(30L)), nullValue());
  }

  @Test
  public void branchesReturnNumbersOfSameType() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "filter: arg < 50",
        "  function: arg",
        "default: arg * 2"
    );

    assertThat(adaptor.getResultType(), is(Types.INTEGER));

    assertThat(adaptor.evaluate(tuple(49L)), is(49L));
    assertThat(adaptor.evaluate(tuple(80L)), is(160L));
  }

  @Test
  public void branchesReturnNumbersOfDifferentType() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "filter: arg < 50",
        "  function: arg",
        "default: arg * 0.5"
    );

    assertThat(adaptor.getResultType(), is(Types.FLOATING));

    assertThat(adaptor.evaluate(tuple(49L)), is(49.0D));
    assertThat(adaptor.evaluate(tuple(80L)), is(40.0D));
  }

  @Test
  public void branchesReturnStructWithNumbersOfSameType() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "filter: arg < 50",
        "  function:",
        "    i: arg",
        "    f: arg * 0.1",
        "default:",
        "  i: arg * 2",
        "  f: arg * 0.2"
    );

    Struct expected = Struct.of("i", Types.INTEGER, "f", Types.FLOATING);
    assertThat(adaptor.getResultType(), is(expected));

    assertThat(adaptor.evaluate(tuple(40L)), is(Tuple.ofValues(expected, 40L, 4.0D)));
    assertThat(adaptor.evaluate(tuple(80L)), is(Tuple.ofValues(expected, 160L, 16.0D)));
  }

  @Test
  public void branchesReturnStructWithNumbersOfDifferenctTypes() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "filter: arg < 50",
        "  function:",
        "    i: arg",
        "    f: arg * 2",
        "default:",
        "  i: arg * 2",
        "  f: arg * 0.2"
    );

    Struct expected = Struct.of("i", Types.INTEGER, "f", Types.FLOATING);
    assertThat(adaptor.getResultType(), is(expected));

    assertThat(adaptor.evaluate(tuple(40L)), is(Tuple.ofValues(expected, 40L, 80.0D)));
    assertThat(adaptor.evaluate(tuple(80L)), is(Tuple.ofValues(expected, 160L, 16.0D)));
  }

  @Test
  public void branchesReturnStructWithDifferentMembersMakesNullableAttributes() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "filter: arg < 50",
        "  function:",
        "    foo: arg",
        "    bar: arg * 2",
        "default:",
        "  foo: arg * 2",
        "  baz: arg * 0.2"
    );

    Struct expected = Struct.of("foo", Types.INTEGER, "baz", Nullable.FLOATING, "bar", Nullable.INTEGER);
    assertThat(adaptor.getResultType(), is(expected));

    assertThat(adaptor.evaluate(tuple(40L)), is(Tuple.ofValues(expected, 40L, null, 80L)));
    assertThat(adaptor.evaluate(tuple(80L)), is(Tuple.ofValues(expected, 160L, 16.0D, null)));
  }

  @Test
  public void branchesReturnStructWithDifferentMembersMakesNullableAttributesNoDefault() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "filter: arg < 50",
        "  function:",
        "    foo: arg",
        "    bar: arg * 2",
        "filter: arg < 70",
        "  function:",
        "    foo: arg * 2",
        "    baz: arg * 0.2"
    );

    Struct expected = Struct.of("foo", Types.INTEGER, "bar", Nullable.INTEGER, "baz", Nullable.FLOATING);
    assertThat(adaptor.getResultType(), is(Nullable.of(expected)));

    assertThat(adaptor.evaluate(tuple(40L)), is(Tuple.ofValues(expected, 40L, 80L, null)));
    assertThat(adaptor.evaluate(tuple(60L)), is(Tuple.ofValues(expected, 120L, null, 12.0D)));
    assertThat(adaptor.evaluate(tuple(80L)), nullValue());
  }

  @Test
  public void branchesReturnTuplesWithAttributesInDifferentOrder() throws Exception {
    //GL165
    build(
        "argument-types:",
        "  arg: integer",
        "filter: arg < 50",
        "  function:{i: arg, f: arg * 2}",
        "default:",
        "  function: {f: arg * 0.2, i: arg * 2}"
    );

    // the default function is the initial best match, hence f then i
    Struct expected = Struct.of("f", Types.FLOATING, "i", Types.INTEGER);
    assertThat(adaptor.getResultType(), is(expected));

    assertThat(adaptor.evaluate(tuple(40L)), is(Tuple.ofValues(expected, 80.0D, 40L)));
    assertThat(adaptor.evaluate(tuple(80L)), is(Tuple.ofValues(expected, 16.0D, 160L)));
  }

  @Test
  public void branchesReturnTuplesWithAttributesInDifferentOrderAndExtraArg() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "filter: arg < 50",
        "  function:{i: arg, f: arg * 2, foo: 'bar'}",
        "default:",
        "  function: {f: arg * 0.2, i: arg * 2}"
    );

    Struct expected = Struct.of("f", Types.FLOATING, "i", Types.INTEGER, "foo", Nullable.TEXT);
    assertThat(adaptor.getResultType(), is(expected));

    assertThat(adaptor.evaluate(tuple(40L)),
        is(Tuple.ofValues(expected, 80.0D, 40L, "bar")));
    assertThat(adaptor.evaluate(tuple(80L)),
        is(Tuple.ofValues(expected, 16.0D, 160L, null)));
  }

  @Test
  public void getBestMatchForSimpleTypes() {
    assertThat(bestMatch(Types.INTEGER, Types.FLOATING), is(Types.FLOATING));
    assertThat(bestMatch(Types.FLOATING, Types.INTEGER), is(Types.FLOATING));

    assertThat(bestMatch(new WithinRange(Types.INTEGER, 0, 10), Types.FLOATING), is(Types.FLOATING));
    assertThat(bestMatch(new WithinRange(Types.FLOATING, 0.0D, 5.0D), Types.INTEGER), is(Types.FLOATING));

    assertThat(bestMatch(Types.INTEGER, Types.TEXT), is(Types.ANYTHING));
    assertThat(bestMatch(Types.FLOATING, Types.TEXT), is(Types.ANYTHING));
    assertThat(bestMatch(Types.INTEGER, Types.GEOMETRY), is(Types.ANYTHING));
    assertThat(bestMatch(Types.FLOATING, Types.GEOMETRY), is(Types.ANYTHING));
    assertThat(bestMatch(Types.TEXT, Types.GEOMETRY), is(Types.ANYTHING));
  }

  @Test
  public void getBestMatchForSimpleNullalbeTypes() {
    assertThat(bestMatch(Nullable.INTEGER, Types.FLOATING), is(Nullable.FLOATING));
    assertThat(bestMatch(Types.FLOATING, Nullable.INTEGER), is(Nullable.FLOATING));

    assertThat(bestMatch(Nullable.INTEGER, Nullable.FLOATING), is(Nullable.FLOATING));
    assertThat(bestMatch(Nullable.FLOATING, Nullable.INTEGER), is(Nullable.FLOATING));

    assertThat(bestMatch(Nullable.of(new WithinRange(Types.INTEGER, 0, 10)), Types.FLOATING), is(Nullable.FLOATING));
    assertThat(bestMatch(Nullable.of(new WithinRange(Types.INTEGER, 0, 10)), Nullable.FLOATING), is(Nullable.FLOATING));
    assertThat(bestMatch(new WithinRange(Types.FLOATING, 0.0D, 5.0D), Nullable.INTEGER), is(Nullable.FLOATING));

    assertThat(bestMatch(Types.INTEGER, Nullable.TEXT), is(Nullable.ANYTHING));
    assertThat(bestMatch(Types.FLOATING, Nullable.TEXT), is(Nullable.ANYTHING));
  }

  @Test
  public void getBestMatchForStructTypes() {
    assertThat(bestMatch(Struct.of("a", Types.INTEGER), Struct.of("a", Types.FLOATING)),
        is(Struct.of("a", Types.FLOATING)));

    assertThat(bestMatch(Struct.of("a", Types.INTEGER), Struct.of("a", Types.FLOATING, "b", Nullable.INTEGER)),
        is(Struct.of("a", Types.FLOATING, "b", Nullable.INTEGER)));

    assertThat(bestMatch(Struct.of("a", Types.INTEGER, "b", Nullable.INTEGER), Struct.of("a", Types.FLOATING)),
        is(Struct.of("a", Types.FLOATING, "b", Nullable.INTEGER)));
  }

  Type bestMatch(Type a, Type b) {
    return ReturnTypeInferer.findBestReceiverType(a, b, project.getTypeSet());
  }

  private Object tuple(Object obj) {
    return Tuple.ofValues(parsed.getInputType(), obj);
  }

}
