/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;


import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.typexp.DefaultTypeBuilder;
import nz.org.riskscape.engine.typexp.TypeBuilder;

public class RealizedTreeFilterTest extends ProjectTest {

  BindingContext context = super.bindingContext;
  ExpressionRealizer realizer = super.expressionRealizer;
  ClassifierFunctionParser parser = new ClassifierFunctionParser();
  TypeBuilder builder = new DefaultTypeBuilder(project.getTypeSet());

  AST.FunctionDecl parsed;
  RealizedTreeFilter filter;

  @Before
  public void before() {
    project.getFunctionSet().insertLast(new DefaultOperators());
  }

  public void parse(String... lines) {
    parsed = parser.parse(Arrays.asList(lines).stream().collect(Collectors.joining("\n")) + "\n");

    parsed.parseTypes(builder);
    parsed.parseExpressions(expressionParser);
  }

  public void build(String... lines) {
    parse(lines);
    filter = RealizedTreeFilter.build(realizer, parsed, parsed.getArgumentTypesDecl().built).orElse(null);
  }

  @Test
  public void aTreeCanBeBuiltFromJustAFunction() throws Exception {
    build("return-type: text",
        "argument-types:",
        "  foo: text\n",
        "function: foo = 'cool'"
    );

    assertNotNull(filter.getOrElse());
    assertEquals(0, filter.getChildren().size());

    assertEquals(Boolean.TRUE, filter.evaluate(Tuple.ofValues(parsed.getInputType(), "cool")));
    assertEquals(Boolean.FALSE, filter.evaluate(Tuple.ofValues(parsed.getInputType(), "hi")));
  }

  @Test
  public void aTreeCanBeBuiltFromASingleFilterAndADefault() throws Exception {
    build(
        "return-type: text",
        "argument-types:",
        "  foo: text",
        "filter: foo = 'cool'",
        "  function: 'yay'",
        "default: 'wah'"
    );

    assertEquals("wah", filter.evaluate(Tuple.ofValues(parsed.getInputType(), "hi")));
    assertEquals("yay", filter.evaluate(Tuple.ofValues(parsed.getInputType(), "cool")));
  }

  @Test
  public void aTreeCanBeBuiltFromAMoreComplicatedSetOfFilters() throws Exception {
    build(
        "return-type: text",
        "argument-types:",
        "  celsius: integer",
        "filter: celsius > 0",
        "  filter: celsius > 20",
        "    filter: celsius >= 100",
        "      function: 'boiling'",
        "    filter: celsius > 90",
        "      function: 'good for coffee'",
        "    filter: celsius < 40",
        "      function: 'have a bath'",
        "    default: 'wash the dishes'",
        "  function: 'make ice cubes'",
        "default: 'freezing'"
    );

    assertEquals("freezing", filter.evaluate(tuple(-20L)));
    assertEquals("freezing", filter.evaluate(tuple(0L)));
    assertEquals("make ice cubes", filter.evaluate(tuple(1L)));
    assertEquals("make ice cubes", filter.evaluate(tuple(20L)));
    assertEquals("have a bath", filter.evaluate(tuple(39L)));
    assertEquals("wash the dishes", filter.evaluate(tuple(40L)));
    assertEquals("wash the dishes", filter.evaluate(tuple(41L)));
    assertEquals("wash the dishes", filter.evaluate(tuple(89L)));
    assertEquals("good for coffee", filter.evaluate(tuple(91L)));
    assertEquals("boiling", filter.evaluate(tuple(100L)));
  }

  private Object tuple(Object obj) {
    return Tuple.ofValues(parsed.getInputType(), obj);
  }
  @Test
  public void resultIsNotNullableIfAllFiltersHaveADefaultCase() throws Exception {
    build(
        "return-type: text",
        "argument-types:",
        "  celsius: integer",
        "default: 'freezing'"
    );
    assertTrue(filter.isDefaultPresent());

    build(
        "return-type: text",
        "argument-types:",
        "  celsius: integer",
        "filter: 1",
        "  function: 1",
        "filter: 2",
        "  function: 2",
        "filter: 3",
        "  function: 3",
        "default: 4"
    );
    assertTrue(filter.isDefaultPresent());
  }

  @Test
  public void givesNullIfNoFiltersMatch() throws Exception {
    build(
        "argument-types:",
        "  celsius: integer",
        "filter: celsius > 50",
        "  function: 'cool'"
    );
    assertFalse(filter.isDefaultPresent());
    assertEquals("cool", filter.evaluate(tuple(51L)));
    assertNull(filter.evaluate(tuple(49L)));
  }

  @Test
  public void resultTypeHasNullableAttributes() throws Exception {
    build(
        "argument-types:",
        "  celsuis: integer",
        "filter: celsuis > 50",
        "  function:",
        "    a: 'cool'",
        "    b: 'foo'",
        "default:",
        "  a: 'unknown'"
    );

    // but the actual results don't, as they are passing the tuple back that was returned by the
    // tree expression.
    assertThat(filter.evaluate(tuple(52L)), is(
        Tuple.ofValues(Struct.of("a", Types.TEXT, "b", Types.TEXT), "cool", "foo")));

    assertThat(filter.evaluate(tuple(30L)), is(
        Tuple.ofValues(Struct.of("a", Types.TEXT), "unknown")));
  }

  @Test
  public void resultTypeHasNullableStructWhenNoDefault() throws Exception {
    build(
        "argument-types:",
        "  celsuis: integer",
        "filter: celsuis > 50",
        "  function:",
        "    a: 'cool'",
        "    b: 'foo'"
    );

    Struct expectedType = Struct.of("a", Types.TEXT, "b", Types.TEXT);

    assertThat(filter.evaluate(tuple(52L)), is(
        Tuple.ofValues(expectedType, "cool", "foo")));

    assertThat(filter.evaluate(tuple(30L)), nullValue());
  }


}
