/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ExpressionParser;

public class RealizedTreeExpressionTest {

  Struct inputType = Struct.of();
  ExpressionRealizer realizer = mock(ExpressionRealizer.class);
  Object result = new Object();

  @Test
  public void canRealizeAndEvaluateASimpleExpression() throws Exception {
    AST.SimpleExpression ast = AST.SimpleExpression.create("foo", ExpressionParser.parseString("ignored"));

    RealizedExpression mockRealized = mock(RealizedExpression.class);
    when(realizer.realize(same(inputType), same(ast.built))).thenReturn(ResultOrProblems.of(mockRealized));
    when(mockRealized.getResultType()).thenReturn(Types.TEXT);
    when(mockRealized.evaluate(any())).thenReturn(result);
    RealizedTreeExpression subject = RealizedTreeExpression.build(realizer, ast, inputType).get();

    assertSame(Types.TEXT, subject.getResultType());
    assertSame(result, subject.evaluate(null));
  }

  @Test
  public void canRealizeAndEvaluateAStructExpression() throws Exception {
    AST.SimpleExpression happy = AST.SimpleExpression.create("happy", ExpressionParser.parseString("are"));
    AST.SimpleExpression days =  AST.SimpleExpression.create("days", ExpressionParser.parseString("here_again"));
    AST.StructExpression ast = AST.StructExpression.create("bar", happy, days);

    RealizedExpression mockHappy = mock(RealizedExpression.class);
    RealizedExpression mockDays = mock(RealizedExpression.class);

    when(realizer.realize(same(inputType), same(happy.built))).thenReturn(ResultOrProblems.of(mockHappy));
    when(mockHappy.evaluate(any())).thenReturn("happy days");
    when(mockHappy.getResultType()).thenReturn(Types.INTEGER);

    when(realizer.realize(same(inputType), same(days.built))).thenReturn(ResultOrProblems.of(mockDays));
    when(mockDays.evaluate(any())).thenReturn("here again");
    when(mockDays.getResultType()).thenReturn(Types.DATE);

    RealizedTreeExpression subject = RealizedTreeExpression.build(realizer, ast, inputType).get();
    assertEquals(
        Struct.of("happy", Types.INTEGER, "days", Types.DATE),
        subject.getResultType()
    );

    Object fooScope = new Object();

    assertEquals(
        Tuple.ofValues((Struct) subject.getResultType(), "happy days", "here again"),
        subject.evaluate(fooScope)
    );
  }


}
