/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import org.junit.Test;

import com.google.common.collect.Sets;

import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.typexp.TypeBuilder;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.Expression;

public class FunctionDeclTest {

  ClassifierFunctionParser parser = new ClassifierFunctionParser();
  TypeBuilder builder = mock(TypeBuilder.class);
  ExpressionParser expressionParser = mock(ExpressionParser.class);

  @Test
  public void canBuildTypesFromAST() {
    AST.FunctionDecl functionDecl = parser.parse("return-type: text\n"
        + "argument-types:\n"
        + "  foo: integer\n"
        + "\n");

    when(builder.build("text")).thenReturn(Types.TEXT);
    when(builder.build("integer")).thenReturn(Types.INTEGER);

    assertEquals(0, functionDecl.parseTypes(builder).size());

    assertEquals(Types.TEXT, functionDecl.returnTypeDecl.get().getBuilt());
    assertEquals(Struct.of("foo", Types.INTEGER), functionDecl.argumentTypesDecl.getBuilt());
  }

  @Test
  public void canBuildExpressionsFromAST() {
    AST.FunctionDecl functionDecl = parser.parse(""
        + "return-type: text\n"
        + "argument-types:\n"
        + "  foo: text\n"
        + "\n"
        + "pre:\n"
        + "  foo: 'bar'\n"
        + "filter: foo > bar\n"
        + "  filter: x = 1\n"
        + "    function: 'cool'\n"
        + "post:\n"
        + "  foo: 0.33\n"
        + ""
        );

    Expression mockBar = mock(Expression.class);
    when(expressionParser.parse("'bar'")).thenReturn(mockBar);
    Expression mockFooGtBar = mock(Expression.class);
    when(expressionParser.parse("foo > bar")).thenReturn(mockFooGtBar);
    Expression mockXeq1 = mock(Expression.class);
    when(expressionParser.parse("x = 1")).thenReturn(mockXeq1);
    Expression mockCool = mock(Expression.class);
    when(expressionParser.parse("'cool'")).thenReturn(mockCool);
    Expression mockZeroThreeThree = mock(Expression.class);
    when(expressionParser.parse("0.33")).thenReturn(mockZeroThreeThree);

    functionDecl.parseExpressions(expressionParser);

    AST.StructExpression pre = assertClass(AST.StructExpression.class, functionDecl.pre.get());
    AST.StructExpression post = assertClass(AST.StructExpression.class, functionDecl.post.get());

    AST.Filter body = assertClass(AST.Filter.class, functionDecl.body.get(0));

    assertEquals(mockBar, assertClass(AST.SimpleExpression.class, pre.find("foo").get()).built);
    assertEquals(mockZeroThreeThree, assertClass(AST.SimpleExpression.class, post.find("foo").get()).built);

    assertSame(mockFooGtBar, body.built);
    AST.Filter child = body.children.get(0);
    assertSame(mockXeq1, child.built);
    assertSame(mockCool, assertClass(AST.SimpleExpression.class, child.orElse.get()).built);
  }

  @Test
  public void canScanAllKeysInStructExpressions() throws Exception {
    AST.FunctionDecl functionDecl = parser.parse(""
        + "return-type: text\n"
        + "argument-types:\n"
        + "  foo: text\n"
        + "\n"
        + "pre:\n"
        + "  foo: 'bar'\n"
        + "  also: good\n"
        + "filter: foo > bar\n"
        + "  filter: x = 1\n"
        + "    function:\n"
        + "      sick: dude\n"
        + "  filter: x = 2\n"
        + "    function:\n"
        + "      brah: son\n"
        + "      totes: cool\n"
        + "post:\n"
        + "  baz: 0.33\n"
        + "  foo: derp\n"
        + ""
        );

    assertEquals(
        Sets.newHashSet("foo", "also", "sick", "brah", "totes", "baz"),
        functionDecl.scanStructKeys()
    );
  }

  private <T> T assertClass(Class<T> clazz, Object toCheck) {
    if (clazz.isInstance(toCheck)) {
      return clazz.cast(toCheck);
    } else {
      fail(toCheck + " not an instance of " + clazz);
      return null;
    }
  }

}
