/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import static nz.org.riskscape.defaults.classifier.ProblemCodes.*;
import static nz.org.riskscape.engine.Assert.*;
import static nz.org.riskscape.engine.Matchers.*;
import static nz.org.riskscape.problem.StandardCodes.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.hamcrest.Description;
import org.hamcrest.Matcher;
import org.hamcrest.TypeSafeMatcher;
import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.FunctionArgument;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.CoercionException;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinRange;
import nz.org.riskscape.engine.types.WithinSet;
import nz.org.riskscape.engine.typexp.DefaultTypeBuilder;
import nz.org.riskscape.engine.typexp.TypeBuilder;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.problem.Problem.Severity;
import nz.org.riskscape.rl.ast.Expression;

@SuppressWarnings("unchecked")
public class ClassifierFunctionTest extends ProjectTest {

  BindingContext context = super.bindingContext;
  ExpressionRealizer realizer = super.expressionRealizer;
  ClassifierFunctionParser parser = new ClassifierFunctionParser();

  TypeBuilder builder = new DefaultTypeBuilder(project.getTypeSet());

  AST.FunctionDecl parsed;
  ClassifierFunction built;
  ResultOrProblems<RiskscapeFunction> realized;
  RiskscapeFunction function;
  private RealizedExpression realizedExpression;

  @Before
  public void before() {
    project.getFunctionSet().insertLast(new DefaultOperators());
    project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);
    project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
  }

  @Test
  public void canExecuteFunctionWithLinkedTypes() throws Exception {
    Type returnType = new WithinRange(Types.FLOATING, -50F, 300F);

    project.getTypeSet().add("temp", Struct.of("celsius", Types.FLOATING));
    project.getTypeSet().add("fahrenheit", returnType);

    buildOnly("argument-types:",
        "  temp: lookup('temp')",
        "return-type: lookup('fahrenheit')",
        "function: (temp.celsius * (9.0 / 5.0)) + 32");

    Struct inputType = built.getArgumentTypes().get(0).find(Struct.class).orElse(null);
    realize(Arrays.asList(inputType));

    assertEquals(212D, evaluate("func(100.0)"));
    assertThat(realized, result(hasReturnType(project.getTypeSet().getLinkedType("fahrenheit"))));
  }

  private Object evaluate(String unparsedExpression) {
    return evaluate(unparsedExpression, Tuple.of(Struct.EMPTY_STRUCT));
  }

  private Object evaluate(String unparsedExpression, Tuple input) {
    ResultOrProblems<RealizedExpression> realizedOr = expressionRealizer.realize(input.getStruct(), unparsedExpression);

    if (!realizedOr.isPresent()) {
      fail(Problem.debugString(realizedOr.getProblems()));
    }

    this.realizedExpression = realizedOr.get();

    return realizedOr.get().evaluate(input);
  }

  @Test
  public void canExecuteASimpleFunctionWithInferredReturnType() throws Exception {
    build("argument-types:",
        "  celsius: floating",
        "function: (celsius * (9.0 / 5.0)) + 32");

    assertThat(realized, result(hasReturnType(Types.FLOATING)));

    assertEquals(212D, (Double) realized.get().call(Arrays.asList(100D)), 0.0001);
  }

  @Test
  public void canExecuteASimpleFunctionWithInferredStructReturnType() throws Exception {
    build("argument-types:",
        "  celsius: floating",
        "function:",
        "  fahrenheit: (celsius * (9.0 / 5.0)) + 32",
        "  kelvin: celsius + 273.15",
        ""
      );

    assertEquals(Struct.of("fahrenheit", Types.FLOATING, "kelvin", Types.FLOATING), function.getReturnType());
    Tuple result = (Tuple) function.call(Arrays.asList(100D));
    assertEquals(212D, (Double) result.fetch("fahrenheit"), 0.0001);
    assertEquals(212D, (Double) result.fetch("kelvin"), 273.15);
  }

  @Test
  public void canExecuteASimpleFunctionWithDeclaredReturnType() throws Exception {
    build(
        "argument-types:",
        "  celsius: floating",
        "return-type: range(floating, 0.0, 100000.0)",
        "function: celsius + 273.15"
    );

    assertEquals(new WithinRange(Types.FLOATING, 0.0D, 100000.0D), function.getReturnType());
    assertEquals(373.15D, (Double) function.call(Arrays.asList(100D)), 0.0001);

    assertThrows(CoercionException.class, () -> function.call(Arrays.asList(-280D)));
  }

  @Test
  public void canExecuteAFunctionWithAPreFunction() throws Exception {
    build(
        "argument-types:",
        "  celsius: floating",
        "pre:",
        "  kelvin: celsius + 273.15",
        "function: round(kelvin / 1000)"
    );

    assertEquals(Types.INTEGER, function.getReturnType());
    assertEquals(3L, function.call(Arrays.asList(3000D)));
  }

  @Test
  public void canExecuteAFunctionWithAPreFunctionAndStructFunction() throws Exception {
    build(
        "argument-types:",
        "  celsius: floating",
        "pre:",
        "  kelvin: celsius + 273.15",
        "function:",
        "  kk: round(kelvin / 1000)"
    );

    assertEquals(Struct.of("kelvin", Types.FLOATING, "kk", Types.INTEGER), function.getReturnType());
    assertEquals(Tuple.ofValues((Struct) function.getReturnType(), 3273.15D, 3L), function.call(Arrays.asList(3000D)));
  }

  @Test
  public void canExecuteAFunctionWithAPreAndPostFunctionAndFunction() throws Exception {
    build(
        "argument-types:",
        "  celsius: floating",
        "pre:",
        "  kelvin: celsius + 273.15",
        "function:",
        "  kk: round(kelvin / 1000)",
        "post: kk"
    );

    assertEquals(Types.INTEGER, function.getReturnType());
    assertEquals(3L, function.call(Arrays.asList(3000D)));
  }

  @Test
  public void canExecuteAFunctionWithQuotedIdentifier() throws Exception {
    build(
        "argument-types:",
        "  celsius: floating",
        "function: \"round\"(celsius)"
    );

    assertEquals(Types.INTEGER, function.getReturnType());
    assertEquals(3L, function.call(Arrays.asList(3.4D)));
    assertEquals(4L, function.call(Arrays.asList(3.6D)));
  }

  @Test
  public void canExecuteAFunctionWithAPreAndPostStructFunctionAndFunction() throws Exception {
    build(
        "argument-types:",
        "  celsius: floating",
        "pre:",
        "  kelvin: celsius + 273.15",
        "function:",
        "  kk: round(kelvin / 1000)",
        "post:",
        "  k1000: kk"
    );

    Struct expectedType = Struct.of("kelvin", Types.FLOATING, "kk", Types.INTEGER, "k1000", Types.INTEGER);

    assertEquals(expectedType, function.getReturnType());
    assertEquals(Tuple.ofValues(expectedType, 3273.15D, 3L, 3L), function.call(Arrays.asList(3000D)));
  }

  @Test
  public void canExecuteAFunctionWithAFilter() throws Exception {
    build(
        "argument-types:",
        "  fahrenheit: floating",
        "pre:",
        "  celsius: (fahrenheit - 32) * (5.0 / 9.0)",
        "filter: celsius > 0",
        "  filter: celsius > 20",
        "    filter: celsius >= 100",
        "      function: 'boiling ' + str(fahrenheit) + 'F'",
        "    filter: celsius > 90",
        "      function: 'good for coffee'",
        "    filter: celsius < 40",
        "      function: 'have a bath'",
        "    default: 'wash the dishes'",
        "  function: 'make ice cubes'",
        "default: 'freezing'"
    );

    assertEquals(Types.TEXT, function.getReturnType());
    assertEquals("freezing", evaluate("func(31.0)"));
    assertEquals("make ice cubes", evaluate("func(40.0)"));
    assertEquals("have a bath", evaluate("func(75.0)"));
    assertEquals("wash the dishes", evaluate("func(120.0)"));
    // integer works as well - gets adapted
    assertEquals("good for coffee", evaluate("func(200)"));
    // so does a single value struct
    assertEquals("boiling 230.0F", evaluate("func({temperature: 230})"));
  }

  @Test
  public void canExecuteAFunctionWithAFilterAndPost() throws Exception {
    build(
        "argument-types:",
        "  fahrenheit: floating",
        "pre:",
        "  celsius: (fahrenheit - 32) * (5.0 / 9.0)",
        "filter: celsius > 0",
        "  filter: celsius > 20",
        "    filter: celsius >= 100",
        "      function: ",
        "        desc: 'boiling ' + str(fahrenheit) + 'F'",
        "    filter: celsius > 90",
        "      function: ",
        "        desc: 'good for coffee'",
        "    filter: celsius < 40",
        "      function:",
        "        desc: 'have a bath'",
        "    default: ",
        "      desc: 'wash the dishes'",
        "  function: ",
        "    desc: 'make ice cubes'",
        "default: ",
        "  desc: 'freezing'",
        "post: desc + ' - ' + str(fahrenheit) + 'F' + ' ' + str(round(celsius)) + 'C'"
    );
    doExecuteAFunctionWithAFilterAndPost();
  }

  @Test
  public void canExecuteAFunctionWithAFilterAndPostNewSyntax() throws Exception {
    build(
        "argument-types:",
        "  fahrenheit: floating",
        "before:",
        "  celsius: (fahrenheit - 32) * (5.0 / 9.0)",
        "when: celsius > 0",
        "  when: celsius > 20",
        "    when: celsius >= 100",
        "      do: ",
        "        desc: 'boiling ' + str(fahrenheit) + 'F'",
        "    when: celsius > 90",
        "      do: ",
        "        desc: 'good for coffee'",
        "    when: celsius < 40",
        "      do:",
        "        desc: 'have a bath'",
        "    default: ",
        "      desc: 'wash the dishes'",
        "  do: ",
        "    desc: 'make ice cubes'",
        "default: ",
        "  desc: 'freezing'",
        "after: desc + ' - ' + str(fahrenheit) + 'F' + ' ' + str(round(celsius)) + 'C'"
    );
    doExecuteAFunctionWithAFilterAndPost();
  }

  protected void doExecuteAFunctionWithAFilterAndPost() throws Exception {

    assertEquals(Types.TEXT, function.getReturnType());
    assertEquals("freezing - 31.0F -1C", evaluate("func(31)"));
  }

  @Test
  public void canExecuteThatSetsTempTuplesInBodyToCrunchInPost() throws Exception {
    build(
        "argument-types:",
        "  input: integer",
        "filter: input > 10",
        "  function:",
        "    tmp: {a: input, b: input * 10}",
        "default:",
        "  function:",
        "    tmp: {a: input, b: input * 100}",
        "post: tmp.a + tmp.b"
    );

    assertEquals(Types.INTEGER, function.getReturnType());

    assertThat(evaluate("func(100)"), is(1100L));
    assertThat(evaluate("func(5)"), is(505L));
  }

  @Test
  public void canExecuteThatSetsManyTempTuplesInBodyToCrunchInPost() throws Exception {
    build(
        "argument-types:",
        "  input: integer",
        "filter: input > 10",
        "  function:",
        "    tmp1: {a: input, b: input * 10}",
        "    tmp2: {a: input, b: input * 10}",
        "default:",
        "  function:",
        "    tmp1: {a: input, b: input * 100}",
        "    tmp2: {a: input, b: input * 100}",
        "post: tmp1.a + tmp2.b"
    );

    assertEquals(Types.INTEGER, function.getReturnType());

    assertThat(function.call(Arrays.asList(100L)), is(1100L));
    assertThat(function.call(Arrays.asList(5L)), is(505L));
  }

  @Test
  public void canExecuteAFunctionWithStructArguments() throws Exception {
    project.getTypeSet().add("foobarInput", Struct.of("foo", Types.TEXT, "bar", Types.TEXT));
    buildOnly(
        "argument-types:",
        "  input: lookup('foobarInput')",
        "    foo: text",
        "    bar: text",
        "default: str_length(input.foo) + str_length(input.bar)"
    );
    Struct inputType = built.getArgumentTypes().get(0).find(Struct.class).orElse(null);
    realize(Arrays.asList(inputType, Types.TEXT, Types.TEXT));

    assertEquals(Types.INTEGER, function.getReturnType());
    assertEquals(6L, function.call(Arrays.asList(Tuple.ofValues(inputType, "baz", "baz"))));
  }

  @Test
  public void canExecuteAFunctionWithInlineStructArguments() throws Exception {
    buildOnly(
        "argument-types:",
        "  input:",
        "    foo: text",
        "    bar: text",
        "default: str_length(input.foo) + str_length(input.bar)"
    );
    Struct inputType = built.getArgumentTypes().get(0).find(Struct.class).orElse(null);
    realize(Arrays.asList(inputType));

    assertEquals(Types.INTEGER, function.getReturnType());
    assertEquals(6L, function.call(Arrays.asList(Tuple.ofValues(inputType, "baz", "baz"))));
  }

  @Test
  public void canExecuteAFunctionWithInlineStructArguments_With_Pre() throws Exception {
    buildOnly(
        "argument-types:",
        "  input:",
        "    foo: text",
        "    bar: text",
        "pre:",
        "  foo_length: str_length(input.foo)",
        "default: foo_length + str_length(input.bar)"
    );
    Struct inputType = built.getArgumentTypes().get(0).find(Struct.class).orElse(null);
    realize(Arrays.asList(inputType));

    assertEquals(Types.INTEGER, function.getReturnType());
    assertEquals(6L, function.call(Arrays.asList(Tuple.ofValues(inputType, "baz", "baz"))));
  }

  @Test
  public void canExecuteAFunctionWithInlineStructArguments_With_Pre_And_Post() throws Exception {
    buildOnly(
        "argument-types:",
        "  input:",
        "    foo: text",
        "    bar: text",
        "pre:",
        "  foo_length: str_length(input.foo)",
        "default:",
        "  bar_length: str_length(input.bar)",
        "post: foo_length + bar_length"
    );
    Struct inputType = built.getArgumentTypes().get(0).find(Struct.class).orElse(null);
    realize(Arrays.asList(inputType));

    assertEquals(Types.INTEGER, function.getReturnType());
    assertEquals(6L, function.call(Arrays.asList(Tuple.ofValues(inputType, "baz", "baz"))));
  }


  @Test
  public void returnTypeCanBeStruct() throws Exception {
    build(
        "argument-types:",
        "  i: integer",
        "default: ",
        "  i: i",
        "  f: float(i)",
        "  x10: i * 10"
    );

    Struct expectedReturnType = Struct.of("i", Types.INTEGER, "f", Types.FLOATING, "x10", Types.INTEGER);
    assertEquals(expectedReturnType, function.getReturnType());

    assertEquals(Tuple.ofValues(expectedReturnType, 10L, 10D, 100L), function.call(Arrays.asList(10L)));
  }

  @Test
  public void returnTypeCanBeStruct_WithNullables() throws Exception {
    build(
        "argument-types:",
        "  i: floating",
        "filter: (i > 10.0) and (i < 20.0)",
        "  function:",
        "    b: 200",
        "filter: i > 20.0",
        "  function:",
        "    a: 20",
        "    b: 100",
        "default: ",
        "  a: 10",
        "post:",
        "  a: a",
        "  b: b"
    );

    //Both a and b will only be when i > 20, so they both become nullable.
    Struct expectedReturnType = Struct.of("a", Nullable.INTEGER, "b", Nullable.INTEGER);
    assertEquals(expectedReturnType, function.getReturnType());

    assertEquals(Tuple.ofValues(expectedReturnType, 20L, 100L), function.call(Arrays.asList(31D)));
    assertEquals(Tuple.ofValues(expectedReturnType, null, 200L), function.call(Arrays.asList(11D)));
    assertEquals(Tuple.ofValues(expectedReturnType, 10L), function.call(Arrays.asList(10D)));
  }

  @Test
  public void returnTypeIsNullable_WhenNoDefault_Struct() throws Exception {
    build(
        "argument-types:",
        "  i: floating",
        "filter: (i > 10.0) and (i < 20.0)",
        "  function:",
        "    b: 200",
        "filter: i > 20.0",
        "  function:",
        "    a: 20",
        "    b: 100"
    );

    //Both a and b will only be when i > 20, so they both become nullable.
    Struct expectedReturnType = Struct.of("b", Types.INTEGER, "a", Nullable.INTEGER);

    Type returnType = function.getReturnType();
    assertTrue(Nullable.is(returnType));
    assertEquals(expectedReturnType, Nullable.strip(returnType));

    assertEquals(Tuple.ofValues(expectedReturnType, 100L, 20L), function.call(Arrays.asList(31D)));
    assertEquals(Tuple.ofValues(expectedReturnType, 200L, null), function.call(Arrays.asList(11D)));
    assertNull(function.call(Arrays.asList(10D)));
  }

  @Test
  public void returnTypeHasAnythingForMixedAttributeTypes() throws Exception {
    build(
        "argument-types:",
        "  i: integer",
        "pre:",
        "  b: i",
        "filter: i < 10",
        "  function:",
        "    a: i * 10",
        "    b: 'ten'",
        "default:",
        "  function:",
        "    a: i"
    );

    Struct expectedReturnType = Struct.of("b", Nullable.ANYTHING, "a", Types.INTEGER);

    assertThat(function.getReturnType(), is(expectedReturnType));

    assertThat(function.call(Arrays.asList(9L)), is(Tuple.ofValues(expectedReturnType, "ten", 90L)));
    assertThat(function.call(Arrays.asList(12L)), is(Tuple.ofValues(expectedReturnType, 12L, 12L)));


  }

  @Test
  public void defaultBlockMayHaveNullFromBody() throws Exception {
    build(
        "argument-types:",
        "  i: floating",
        "filter: (i > 10.0) and (i < 20.0)",
        "  function:",
        "    b: 200",
        "filter: i > 20.0",
        "  function:",
        "    a: 20",
        "    b: 100",
        "post:",
        "  a: a + 10"
    );

    //Both a and b will only be when i > 20, so they both become nullable.
    Struct expectedReturnType = Struct.of("b", Nullable.INTEGER, "a", Nullable.INTEGER);

    assertEquals(expectedReturnType, function.getReturnType());

    assertEquals(Tuple.ofValues(expectedReturnType, 100L, 30L), function.call(Arrays.asList(31D)));
    assertEquals(Tuple.ofValues(expectedReturnType, 200L), function.call(Arrays.asList(11D)));
    assertEquals(Tuple.ofValues(expectedReturnType), function.call(Arrays.asList(9D)));
  }

  @Test
  public void canHaveManyTopLevelFilters() throws Exception {
    build(
        "argument-types:",
        "  i: integer",
        "filter: i < 20",
        "  function:",
        "    a: 200",
        "filter: i < 50",
        "  function:",
        "    a: 500",
        "default: ",
        "  a: 10",
        "post:",
        "  a: a"
    );

    Struct expectedReturnType = Struct.of("a", Types.INTEGER);
    assertEquals(expectedReturnType, function.getReturnType());

    assertEquals(Tuple.ofValues(expectedReturnType, 200L), function.call(Arrays.asList(-100L)));
    assertEquals(Tuple.ofValues(expectedReturnType, 200L), function.call(Arrays.asList(19L)));

    assertEquals(Tuple.ofValues(expectedReturnType, 500L), function.call(Arrays.asList(20L)));
    assertEquals(Tuple.ofValues(expectedReturnType, 500L), function.call(Arrays.asList(49L)));

    assertEquals(Tuple.ofValues(expectedReturnType, 10L), function.call(Arrays.asList(50L)));
  }

  @Test
  public void canSwapOrderingOfStructAttributes() throws Exception {
    build(
        "argument-types:",
        "  i: floating",
        "filter: i > 20.0",
        "  function:",
        "    b: 100",
        "    a: 20",
        "filter: i > 10.0",
        "  function:",
        "    a: 10",
        "    b: 100",
        "default: ",
        "  a: 10",
        "post:",
        "  a: a",
        "  b: b"  // b will only contain a value when i > 20.0
    );

    Struct expectedReturnType = Struct.of("a", Types.INTEGER, "b", Nullable.INTEGER);
    assertEquals(expectedReturnType, function.getReturnType());

    assertEquals(Tuple.ofValues(expectedReturnType, 20L, 100L), function.call(Arrays.asList(31D)));
    assertEquals(Tuple.ofValues(expectedReturnType, 10L, 100L), function.call(Arrays.asList(15D)));
    assertEquals(Tuple.ofValues(expectedReturnType, 10L), function.call(Arrays.asList(10D)));
  }

  @Test
  public void failsToBuildIfOnlyAPreSection() throws Exception {
    build(
        "argument-types:",
        "  fahrenheit: floating",
        "pre:",
        "  celsius: fahrenheit");

    assertThat(
        realized,
        failedResult(
            isProblem(Severity.ERROR, EMPTY_BODY)
    ));
  }

  @Test
  public void preMustYieldStructReturnType() throws Exception {
    build(
        "argument-types:",
        "  fahrenheit: floating",
        "pre: 'bar'",
        "function: 'foo'"
    );

    assertThat(realized,
        failedResult(isProblem(
            Severity.ERROR,
            PRE_NOT_STRUCT
    )));
  }

  @Test
  public void bodyMustYieldStructTypeIfPostGiven() throws Exception {
    build(
        "argument-types:",
        "  fahrenheit: floating",
        "function: 'foo'",
        "post: 'baz'"
    );

    assertThat(realized,
        failedResult(isProblem(
            Severity.ERROR,
            BODY_NOT_STRUCT
    )));
  }

  @Test
  public void willNotRealizeIfExpressionsFailToRealize() throws Exception {
    build(
        "argument-types:",
        "  fahrenheit: floating",
        "pre:",
        "  celsius: fahrenheit",
        "function:",
        "  result: kelvin");
    assertThat(
        realized,
        failedResult(
            isProblemAffectingLine(5, "function"),
            Matchers.hasProblems(
                isProblemAffectingLine(6, "result")
              )
        )
    );

    build(
        "argument-types:",
        "  fahrenheit: floating",
        "pre:",
        "  celsius: fahrenheit",
        "function:",
        "  kelvin: celsius + 273.15",
        "post: {f: fahrenheit, c: celsius, k: kelvin, why: not}");

    assertThat(
        realized,
        failedResult(
            isProblemAffectingLine(7, "post")
        )
    );
  }

  @Test
  public void canReturnSingleValueStruct_WhenSimpleTypeIsProduced() throws Exception {
    build(
        "argument-types:",
        "  i: integer",
        "return-type:",
        "  a: integer",
        "default:",
        "  function: i * 10");


    assertThat(realized.getProblems(), empty());
    Struct expectedReturnType = Struct.of("a", Types.INTEGER);
    assertEquals(expectedReturnType, function.getReturnType());
    assertEquals(Tuple.ofValues(expectedReturnType, 50L), function.call(Arrays.asList(5L)));
  }

  @Test
  public void returnTypeMustBeAStructWhenLastSectionReturnsAStruct() throws Exception {
    buildOnly(
        "argument-types:",
        "  foo: struct(foo: text)",
        "return-type: text",
        "default: foo"
    );
    Struct inputType = built.getArgumentTypes().get(0).find(Struct.class).orElse(null);
    realize(Arrays.asList(inputType));

    assertThat(realized.getProblems(), contains(
        isProblemShown(Severity.ERROR, containsString("Type mismatch for 'return-type' on line 3"))
      ));
  }

  @Test
  public void missingNonNullableStructMembersInReturnTypeGivesAnError() throws Exception {
    buildOnly(
        "argument-types:",
        "  foo: struct(foo: text)",
        "return-type: struct(foo: text, bar: nullable(text))",
        "default: {}"
    );

    Struct inputType = built.getArgumentTypes().get(0).find(Struct.class).orElse(null);
    realize(Arrays.asList(inputType));

    assertThat(
        realized.getProblems(),
        contains(
            allOf(
              isProblemAffectingLine(3, "return-type"),
              hasProblems(
                    isProblem(Severity.ERROR, ProblemCodes.RETURN_TYPE_MISMATCH)
              )
            )
          )
        );
    // the root cause should be that the foo struct member is missing from the body
    Problem root = realized.filterProblems(StructMember.class).get(0);
    assertThat(root.getAffected(StructMember.class), is(inputType.getMember("foo")));
  }

  @Test
  public void returnTypeCanBeMoreSpecificThanImpliedType() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "return-type: range(integer, 0, 999)",
        "default: arg"
    );

    assertThat(realized.getProblems(), empty());

    assertEquals(1L, function.call(Arrays.asList(1L)));
    assertEquals(new WithinRange(Types.INTEGER, 0, 999), function.getReturnType());
    assertThrows(CoercionException.class, () -> function.call(Arrays.asList(-1L)));
  }

  @Test
  public void returnTypeStructMembersCanBeMoreSpecificThanImpliedType() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "",
        "return-type: struct(foo: range(integer, 0, 999))",
        "function: ",
        "  foo: arg"
    );

    assertThat(realized.getProblems(), empty());

    Type returnType = function.getReturnType();
    assertEquals(Struct.of("foo", new WithinRange(Types.INTEGER, 0, 999)), returnType);
    Tuple result = (Tuple) function.call(Arrays.asList(1L));
    assertEquals(Tuple.ofValues((Struct) returnType, 1L), result);
    assertSame(returnType, result.getStruct());

    assertThrows(CoercionException.class, () -> function.call(Arrays.asList(-1L)));
  }

  @Test
  public void extraStructMembersFromInferredTypeAreIgnoredIfNotInDeclaredType() throws Exception {
    build(
        "argument-types:",
        "  arg: integer",
        "return-type: struct(foo: integer)",
        "function: ",
        "  foo: arg",
        "  bar: arg * 2.0"
    );

    assertThat(realized.getProblems(), empty());

    Type returnType = function.getReturnType();
    assertEquals(Struct.of("foo", Types.INTEGER), returnType);
    Tuple result = (Tuple) function.call(Arrays.asList(1L));
    assertEquals(Tuple.ofValues((Struct) returnType, 1L), result);
    assertSame(returnType, result.getStruct());
  }

  @Test
  public void declaredReturnTypeIsWrappedWithNullableIfInferredTypeIsNullable() throws Exception {
    build(Arrays.asList(Nullable.INTEGER),
        "argument-types:",
        "  arg: nullable(integer)",
        "return-type: integer",
        "function: arg"
    );

    assertThat(realized.getProblems(), empty());
    assertThat(function.getReturnType(), equalTo(Nullable.INTEGER));
    assertThat(function.call(Arrays.asList()), equalTo(null));

  }
  @Test
  public void declaredStructReturnTypeIsWrappedWithNullableIfInferredTypeIsNullable() throws Exception {
    build(
        Arrays.asList(Nullable.of(Struct.of("foo", Types.INTEGER))),
        "argument-types:",
        "  arg: nullable(struct(foo: integer))",
        "return-type: struct(foo: integer)",
        "function: arg"
    );

    assertThat(realized.getProblems(), empty());
    assertThat(function.getReturnType(), equalTo(Nullable.of(Struct.of("foo", Types.INTEGER))));
  }

  @Test
  public void badFilterProblemsComeBackDuringRealization() throws Exception {
    buildOnly(
        "argument-types:",
        "  arg: struct(foo: integer)",
        "filter: arg.baz",
        "  function: arg.foo * 10"
    );
    Struct inputType = built.getArgumentTypes().get(0).find(Struct.class).orElse(null);
    realize(Arrays.asList(inputType));

    assertThat(
        realized,
        Matchers.failedResult(
            Matchers.isProblemAffectingLine(3, "filter"),
            Matchers.hasProblems(
                Matchers.isProblemAffecting(Severity.ERROR, Expression.class)
            )
        )
    );
  }

  @Test
  public void badExpressionProblemsComeBackDuringRealization() throws Exception {
    buildOnly(
        "argument-types:",
        "  arg: struct(foo: integer)",
        "filter: arg.foo",
        "  function: arg.bar * 10"
    );
    Struct inputType = built.getArgumentTypes().get(0).find(Struct.class).orElse(null);
    realize(Arrays.asList(inputType));

    assertThat(
        realized,
        Matchers.failedResult(
            Matchers.isProblemAffectingLine(3, "filter"),
            Matchers.hasProblems(
                Matchers.isProblemAffectingLine(4, "function")
            )
        )
    );
  }


  @Test
  public void canGetTypeInfoWithoutRealization() throws Exception {
    parse(
        "argument-types:",
        "  foo: text",
        "  bar: integer",
        "return-type: floating"
    );

    built = new ClassifierFunction(parsed, project);
    assertEquals(
        built.getArguments(),
        ArgumentList.fromArray(
            new FunctionArgument("foo", Types.TEXT),
            new FunctionArgument("bar", Types.INTEGER)
        )
    );

    assertEquals(built.getReturnType(), Types.FLOATING);
  }

  @Test
  public void emptyFilterIsAProblem() {
    build(
        "argument-types:",
        "  arg1: integer",
        "filter: arg1 > 10",
        "default:",
        "  function: 100",
        ""
    );

    assertThat(realized, failedResult(
        isProblem(Severity.ERROR, PROBLEMS_FOUND, contains(
            isProblem(Severity.ERROR, EMPTY_FILTER)
        ))
    ));
  }

  @Test
  public void bungArgumentTypesResultsInAnything() throws Exception {
    parse(
        "argument-types:",
        "  foo: bogus",
        "  bar: integer",
        "return-type: dodgy"
    );

    built = new ClassifierFunction(parsed, project);
    assertEquals(
        built.getArguments(),
        ArgumentList.fromArray(
            new FunctionArgument("foo", Types.ANYTHING),
            new FunctionArgument("bar", Types.INTEGER)
        )
    );

    assertEquals(Types.ANYTHING, built.getReturnType());
  }

  @Test
  public void lookupUpUnknownTypesIsCaughtAtValidation() throws Exception {
    parse(
        "argument-types:",
        "  foo: lookup('unknown_type')",
        "  bar: integer",
        "function: 'some value"
    );
    built = new ClassifierFunction(parsed, project);

    assertThat(built.validate(realizationContext), failedResult(
        isError(GeneralProblems.class, "noSuchObjectExists")
    ));
  }

  @Test
  public void canBeRealizedAgainstCovariantTypes() throws Exception {
    parse(
        "argument-types:",
        "  foo: text",
        "function: 'foo'"
    );
    built = new ClassifierFunction(parsed, project);

    project.getFunctionSet().add(built.identified("foo"));
    Struct inputType = Struct.of("foo", new WithinSet(Types.TEXT, "foo", "bar"));
    RealizedExpression expr = realizationContext.getExpressionRealizer()
      .realize(inputType, "foo('bar')").get();

    assertEquals("foo", expr.evaluate(Tuple.ofValues(inputType, "bar")));
  }

  @Test
  public void cannotMixSimpleAndComplexResultTypes() throws Exception {
    build(Arrays.asList(Types.INTEGER),
        "argument-types:",
        "  arg: integer",
        "filter: arg > 10",
        "  function:",
        "    times2: arg * 2",
        "    times3: arg * 3",
        "default: arg + 10");

    assertThat(realized.getProblems(), contains(
        isProblem(Severity.ERROR, ProblemCodes.MIXED_TREE_RESULT_TYPES)
    ));
  }

  @Test
  public void canBeRealizedAgainstCoerceableTypes() throws Exception {
    parse(
        "argument-types:",
        "  temp: struct(description: text)",
        "function: temp.description"
    );
    built = new ClassifierFunction(parsed, project);

    project.getFunctionSet().add(built.identified("foo"));
    Struct inputType = Struct.of();
    ResultOrProblems<RealizedExpression> exprOr = realizationContext.getExpressionRealizer()
      .realize(inputType, "foo('hot')");
    RealizedExpression expr = exprOr.orElse(null);

    assertEquals(Types.TEXT, expr.getResultType());
    assertEquals("hot", expr.evaluate(Tuple.ofValues(Struct.EMPTY_STRUCT)));
  }

  public void parse(String... lines) {
    parsed = parser.parse(Arrays.asList(lines).stream().collect(Collectors.joining("\n")) + "\n");
  }

  public void build(String... lines) {
    build(Collections.emptyList(), lines);
  }

  public void build(List<Type> argTypes, String... lines) {
    buildOnly(lines);
    realize(argTypes);
  }

  public void buildOnly(String... lines) {
    parse(lines);
    built = new ClassifierFunction(parsed, project);
    project.getFunctionSet().remove("func");
    project.getFunctionSet().add(built.identified("func"));
  }

  public void realize(List<Type> argTypes) {
    realized = built.realize(realizationContext, null, argTypes);
    function = realized.orElse(null);
  }

  private Matcher<RiskscapeFunction> hasReturnType(Type expectedReturnType) {
    return new TypeSafeMatcher<RiskscapeFunction>(RiskscapeFunction.class) {

      @Override
      public void describeTo(Description description) {
        description.appendText("function returning ").appendValue(expectedReturnType);
      }

      @Override
      protected void describeMismatchSafely(RiskscapeFunction item, Description mismatchDescription) {
        mismatchDescription.appendText("function ").appendValue(item).appendText(" returned ")
          .appendValue(item.getReturnType());
      }

      @Override
      protected boolean matchesSafely(RiskscapeFunction item) {
        return item.getReturnType().equals(expectedReturnType);
      }
    };
  }
}
