/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import static nz.org.riskscape.defaults.classifier.ProblemCodes.*;
import static nz.org.riskscape.engine.Assert.*;
import static nz.org.riskscape.engine.Matchers.*;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

import org.junit.Test;

import nz.org.riskscape.dsl.InconsistentIndentException;
import nz.org.riskscape.dsl.Lexer;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.problem.Problem.Severity;

public class ClassifierFunctionParserTest {

  private static final String CR_LF = "\r\n";
  ClassifierFunctionParser parser = new ClassifierFunctionParser();
  AST.FunctionDecl parsed;

  @Test
  public void canParseTheSimplestFunction() throws Exception {
    parse(
        "return-type: return-this",
        "argument-types:",
        "  arg1: arg-value",
        "function: my function expression",
        ""
    );
    doParseTheSimplestFunction();
  }

  @Test
  public void canParseTheSimplestFunctionWithNewSyntax() throws Exception {
    parse(
        "return-type: return-this",
        "argument-types:",
        "  arg1: arg-value",
        "do: my function expression",
        ""
    );
    doParseTheSimplestFunction();
  }

  protected void doParseTheSimplestFunction() throws Exception {

    assertEquals("return-this", assertClass(AST.SimpleType.class, parsed.returnTypeDecl.get()).expression.value);

    assertEquals("arg-value",
        assertClass(AST.SimpleType.class, parsed.argumentTypesDecl.find("arg1").get()).expression.value);

    assertEquals("my function expression",
        assertClass(AST.SimpleExpression.class, parsed.defaultExpr.get()).expression);
  }

  @Test
  public void functionsCanHaveComments() throws Exception {
    parse(
        "return-type: return-this # this is great",
        "argument-types:",
        "  # this one is scary",
        "  arg1: arg-value",
        "function: my function expression",
        "# any more?"
    );

    assertEquals("return-this", assertClass(AST.SimpleType.class, parsed.returnTypeDecl.get()).expression.value);

    assertEquals("arg-value",
        assertClass(AST.SimpleType.class, parsed.argumentTypesDecl.find("arg1").get()).expression.value);

    assertEquals("my function expression",
        assertClass(AST.SimpleExpression.class, parsed.defaultExpr.get()).expression);
  }


  @Test
  public void canParseTheSimplestFunctionWithDefaultAlternative() throws Exception {
    parse(
        "return-type: return-this",
        "argument-types:",
        "  arg1: arg-value",
        "default: my function expression",
        ""
    );

    assertEquals("return-this", assertClass(AST.SimpleType.class, parsed.returnTypeDecl.get()).expression.value);

    assertEquals("arg-value",
        assertClass(AST.SimpleType.class, parsed.argumentTypesDecl.find("arg1").get()).expression.value);

    assertEquals("my function expression",
        assertClass(AST.SimpleExpression.class, parsed.defaultExpr.get()).expression);
  }


  @Test
  public void canParseAComplexReturnType() throws Exception {
    parse(
        "return-type: ",
        "  return-a: is-a",
        "  return-b: is-not-a",
        "argument-types:",
        "  arg1: arg-value",
        "function: my function expression",
        ""
    );

    AST.StructType returnType = assertClass(AST.StructType.class, parsed.returnTypeDecl.get());

    assertEquals("is-a", assertClass(AST.SimpleType.class, returnType.find("return-a").get()).expression.value);
    assertEquals("is-not-a", assertClass(AST.SimpleType.class, returnType.find("return-b").get()).expression.value);

    assertEquals("arg-value",
        assertClass(AST.SimpleType.class, parsed.argumentTypesDecl.find("arg1").get()).expression.value);

    assertEquals("my function expression",
        assertClass(AST.SimpleExpression.class, parsed.defaultExpr.get()).expression);
  }

  @Test
  public void canParseAFunctionWithPreAndPost() throws Exception {
    parse(
        "return-type: text",
        "argument-types:",
        "  arg1: arg-value",
        "pre:",
        "  start: 'it begins'",
        "function:",
        "  middle: funky",
        "post:",
        "  finish: 'it ends'",
        ""
    );
    doParseAFunctionWithPreAndPost();
  }

  @Test
  public void canParseAFunctionWithBeforeAndAfter() throws Exception {
    parse(
        "return-type: text",
        "argument-types:",
        "  arg1: arg-value",
        "before:",
        "  start: 'it begins'",
        "do:",
        "  middle: funky",
        "after:",
        "  finish: 'it ends'",
        ""
    );
    doParseAFunctionWithPreAndPost();
  }

  private void doParseAFunctionWithPreAndPost() {

    assertEquals("text", assertClass(AST.SimpleType.class, parsed.returnTypeDecl.get()).expression.value);

    assertEquals("arg-value",
        assertClass(AST.SimpleType.class, parsed.argumentTypesDecl.find("arg1").get()).expression.value);

    AST.StructExpression pre = assertClass(AST.StructExpression.class, parsed.pre.get());
    AST.StructExpression function = assertClass(AST.StructExpression.class, parsed.defaultExpr.get());
    AST.StructExpression post = assertClass(AST.StructExpression.class, parsed.post.get());

    assertEquals("'it begins'", assertClass(AST.SimpleExpression.class, pre.find("start").get()).expression);
    assertEquals("funky", assertClass(AST.SimpleExpression.class, function.find("middle").get()).expression);
    assertEquals("'it ends'", assertClass(AST.SimpleExpression.class, post.find("finish").get()).expression);
  }


  @Test
  public void canParseAFunctionWithASingleFilter() throws Exception {
    parse(
        "return-type: text",
        "argument-types:",
        "  arg1: arg-value",
        "filter: foo = bar",
        "  function: 'cool'",
        "default: 'rad'",
        ""
    );
    doParseAFunctionWithASingleFilter();
  }

  @Test
  public void canParseAFunctionWithASingleFilterNewSyntax() throws Exception {
    parse(
        "return-type: text",
        "argument-types:",
        "  arg1: arg-value",
        "when: foo = bar",
        "  do: 'cool'",
        "default: 'rad'",
        ""
    );
    doParseAFunctionWithASingleFilter();
  }

  protected void doParseAFunctionWithASingleFilter() throws Exception {

    AST.Filter body = assertClass(AST.Filter.class, parsed.body.get(0));
    AST.SimpleExpression filterFunction = assertClass(AST.SimpleExpression.class, body.orElse.get());
    AST.SimpleExpression function = assertClass(AST.SimpleExpression.class, parsed.defaultExpr.get());

    assertEquals("foo = bar", body.filterExpression.value);
    assertEquals("'cool'", filterFunction.expression);
    assertEquals("'rad'", function.expression);
  }

  @Test
  public void canParseAFunctionWithANestedFilter() throws Exception {
    parse(
        "return-type: text",
        "argument-types:",
        "  arg1: arg-value",
        "filter: foo = bar",
        "  filter: foo = baz",
        "    function: 'story'",
        "  filter: foo = oof",
        "    function: 'bro'",
        "  function: 'cool'",
        "default: 'rad'",
        ""
    );
    doParseAFunctionWithANestedFilter();
  }

  @Test
  public void canParseAFunctionWithANestedFilterNewSyntax() throws Exception {
    parse(
        "return-type: text",
        "argument-types:",
        "  arg1: arg-value",
        "when: foo = bar",
        "  when: foo = baz",
        "    do: 'story'",
        "  when: foo = oof",
        "    do: 'bro'",
        "  do: 'cool'",
        "default: 'rad'",
        ""
    );
    doParseAFunctionWithANestedFilter();
  }

  @Test
  public void canParseAFunctionWithANestedFilterMixedSyntax() throws Exception {
    // it probably doesn't make sense to mix up the syntax but it does work fine.
    parse(
        "return-type: text",
        "argument-types:",
        "  arg1: arg-value",
        "filter: foo = bar",
        "  when: foo = baz",
        "    function: 'story'",
        "  filter: foo = oof",
        "    do: 'bro'",
        "  do: 'cool'",
        "default: 'rad'",
        ""
    );
    doParseAFunctionWithANestedFilter();
  }

  protected void doParseAFunctionWithANestedFilter() throws Exception {

    AST.Filter body = assertClass(AST.Filter.class, parsed.body.get(0));
    AST.SimpleExpression filterFunction = assertClass(AST.SimpleExpression.class, body.orElse.get());
    AST.SimpleExpression function = assertClass(AST.SimpleExpression.class, parsed.defaultExpr.get());
    assertEquals("foo = bar", body.filterExpression.value);
    assertEquals("'cool'", filterFunction.expression);
    assertEquals("'rad'", function.expression);

    AST.Filter child0 = body.children.get(0);
    assertEquals("foo = baz", child0.filterExpression.value);
    assertEquals("'story'", assertClass(AST.SimpleExpression.class, child0.orElse.get()).expression);

    AST.Filter child1 = body.children.get(1);
    assertEquals("foo = oof", child1.filterExpression.value);
    assertEquals("'bro'", assertClass(AST.SimpleExpression.class, child1.orElse.get()).expression);
  }

  @Test
  public void canParse_id_and_desc() {
    parse(
        "id: test",
        "description: test description",
        "category: misc",
        "default: 5"
    );
    assertThat(parsed.id.get().value.getValue(), is("test"));
    assertThat(parsed.description.get().value.getValue(), is("test description"));
  }

  @Test
  public void canParse_quoted_id_and_desc() {
    parse(
        "id: \"my:test\"",
        "description: \"my:test description\"",
        "category: misc",
        "default: 5"
    );
    assertThat(parsed.id.get().value.getValue(), is("my:test"));
    assertThat(parsed.description.get().value.getValue(), is("my:test description"));
  }

  @Test
  public void canParse_id_with_double_colon_and_desc() {
    parse(
        "id: my::test",
        "description: my::test description",
        "category: misc",
        "default: 5"
    );
    assertThat(parsed.id.get().value.getValue(), is("my::test"));
    assertThat(parsed.description.get().value.getValue(), is("my::test description"));
  }


  @Test
  public void canParseWithWindowsLineEndings() throws Exception {

    List<String> source = Arrays.asList(
        "id: test",
        "description: descriptino",
        "category: misc",
        "filter: true",
        "  filter: false",
        "    function: cool",
        "  function: bar"
    );


    List<Token> tokensWindows = lexAll(source.stream().collect(Collectors.joining(CR_LF)));
    List<Token> tokensUnix = lexAll(source.stream().collect(Collectors.joining("\n")));

    assertEquals(tokensWindows.size(), tokensUnix.size());
    Iterator<Token> tokensWinIter = tokensWindows.iterator();
    for (Token token : tokensUnix) {
      Token winToken = tokensWinIter.next();
      assertEquals(token.value, winToken.value);
      assertEquals(token.type, winToken.type);
      assertEquals(token.getLocation().getLine(), winToken.getLocation().getLine());
      assertEquals(token.countIndents(), winToken.countIndents());
    }
  }

  @Test
  public void mixedIndentingIsOkAsLongAsItIsConsistent() throws Exception {
    parse(
        "return-type: text",
        "argument-types:",
        "\targ1: arg-value",
        "filter: foo = bar",
        "  filter: foo = baz",
        "    function: 'story'",
        "  filter: foo = oof",
        "  \tfunction: 'bro'",
        "  function: 'cool'",
        "default: 'rad'",
        ""
    );

    AST.Filter body = assertClass(AST.Filter.class, parsed.body.get(0));
    AST.SimpleExpression filterFunction = assertClass(AST.SimpleExpression.class, body.orElse.get());
    AST.SimpleExpression function = assertClass(AST.SimpleExpression.class, parsed.defaultExpr.get());
    assertEquals("foo = bar", body.filterExpression.value);
    assertEquals("'cool'", filterFunction.expression);
    assertEquals("'rad'", function.expression);

    AST.Filter child0 = body.children.get(0);
    assertEquals("foo = baz", child0.filterExpression.value);
    assertEquals("'story'", assertClass(AST.SimpleExpression.class, child0.orElse.get()).expression);

    AST.Filter child1 = body.children.get(1);
    assertEquals("foo = oof", child1.filterExpression.value);
    assertEquals("'bro'", assertClass(AST.SimpleExpression.class, child1.orElse.get()).expression);
  }

  @Test(expected = InconsistentIndentException.class)
  public void mixedInconsistentIndentingIsNotOk() throws Exception {
    parse(
        "return-type: text",
        "argument-types:",
        "\targ1: arg-value",
        "filter: foo = bar",
        "\t filter: foo = baz",
        "    function: 'story'",
        "  filter: foo = oof",
        "  \tfunction: 'bro'",
        "  function: 'cool'",
        "default: 'rad'",
        ""
    );
  }

  @Test
  public void invalidIdentifierIsAProblem() {
    UnexpectedIdentifierException ex = assertThrows(UnexpectedIdentifierException.class, () -> parse(
        "return-type: return-this",
        "argument-types:",
        "  arg1: arg-value",
        "doit: my function expression",
        ""));
  }

  @Test
  public void redundantFunctionIsAProblem() {
    // a function (outside of filter) becomes a default
    // but if there is a default it would be ignored. Hence the invalid syntax exception.
    InvalidClassifierSyntaxException ex = assertThrows(InvalidClassifierSyntaxException.class, () -> parse(
        "argument-types:",
        "  arg1: integer",
        "function: arg1 * 10",
        "default:",
        "  function: 100",
        ""
    ));

    assertThat(ex.getProblem(), isProblem(Severity.ERROR, FUNCTION_AND_DEFAULT));
  }

  @Test
  public void cannotRedefineFunctionParameters() {
    InvalidClassifierSyntaxException ex = assertThrows(InvalidClassifierSyntaxException.class, () -> parse(
        "argument-types:",
        "  arg1: integer",
        "argument-types:",
        "  arg1: integer",
        "default:",
        "  function: 100",
        ""
    ));

    assertThat(ex.getProblem(), isProblem(Severity.ERROR, REDEFINITION));
  }

  private List<Token> lexAll(String source) {
    Lexer<TokenTypes> lexer = new Lexer<>(TokenTypes.tokens(), source);
    List<Token> tokens = new ArrayList<>();

    while (!lexer.isEOF()) {
      tokens.add(lexer.next());
    }

    return tokens;
  }

  private <T> T assertClass(Class<T> clazz, Object toCheck) {
    if (clazz.isInstance(toCheck)) {
      return clazz.cast(toCheck);
    } else {
      fail(toCheck + " not an instance of " + clazz);
      return null;
    }
  }

  private void parse(String... lines) {
    String source = Arrays.asList(lines).stream().collect(Collectors.joining("\n"));
    this.parsed = parser.parse(source);
  }
}
