/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.defaults.classifier;


import static nz.org.riskscape.engine.Matchers.*;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.Arrays;
import java.util.stream.Collectors;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.StringFunctions;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.engine.rl.DefaultOperators;
import nz.org.riskscape.engine.rl.MathsFunctions;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;

/**
 * Integration test for classifier functions and how they work with the function resolver.
 */
public class ClassifierFunctionIntegrationTest extends BaseExpressionRealizerTest {

  ClassifierFunctionParser parser = new ClassifierFunctionParser();
  Struct argumentType = Struct.of("attr", Types.TEXT, "number", Types.FLOATING);
  Struct inputType = Struct.of("value", argumentType, "value2", Nullable.of(argumentType));

  @Before
  public void addFunctions() {
    project.getFunctionSet().insertFirst(DefaultOperators.INSTANCE);
    project.getFunctionSet().addAll(MathsFunctions.FUNCTIONS);
    project.getFunctionSet().addAll(StringFunctions.FUNCTIONS);
    addEchoFunction();
    addSumFunction();
  }

  void addEchoFunction() {
    build("echo",
        "argument-types:",
        " input:",
        "   attr: text",
        "default:",
        " function: input.attr");
  }

  void addSumFunction() {
    build("sum",
        "argument-types:",
        " arg1:",
        "   number: floating",
        " arg2:",
        "   number: floating",
        "default:",
        " function: arg1.number + arg2.number");
  }

  @Test
  public void canUseInputOfRequiredType() {
    Tuple input = Tuple.ofValues(inputType, Tuple.ofValues(argumentType, "boom", 3D));
    realize(inputType, parse("echo(value.attr)"));
    assertThat(realizationProblems, empty());
    assertThat(realized.evaluate(input), equalTo("boom"));
    assertThat(realized.getResultType(), equalTo(Types.TEXT));
  }

  @Test
  public void canUseStructInputWithRequiredMembers() {
    Tuple input = Tuple.ofValues(inputType, Tuple.ofValues(argumentType, "boom", 3D));
    realize(inputType, parse("echo(value)"));
    assertThat(realizationProblems, empty());
    assertThat(realized.evaluate(input), equalTo("boom"));
    assertThat(realized.getResultType(), equalTo(Types.TEXT));
  }

  @Test
  public void canUseNullableStructInputWithRequiredMembers() {
    Tuple input = Tuple.ofValues(inputType, Tuple.ofValues(argumentType, "boom", 3D),
        Tuple.ofValues(argumentType, "boom boom", 3D));
    realize(inputType, parse("echo(value2)"));
    assertThat(realizationProblems, empty());
    assertThat(realized.evaluate(input), equalTo("boom boom"));
    assertThat(realized.getResultType(), equalTo(Nullable.TEXT));
  }

  @Test
  public void canUseOneNullableStructWithRequiredMembers() {
    // Testing that function resolving works for nullable(struct) -> struct(same attributes).
    // Previously the Nullable wrapping casueed the variance rules to return an equals match but
    // in this case the structs of same attributes need coercing.
    Tuple input = Tuple.ofValues(inputType, Tuple.ofValues(argumentType, "boom", 3.2D),
        Tuple.ofValues(argumentType, "boom boom", 2D));
    // value2 is nullable
    realize(inputType, parse("sum(value, value2)"));
    assertThat(realizationProblems, empty());
    // Now evaluate to ensure no not my struct errors.
    assertThat(realized.evaluate(input), equalTo(5.2D));
    assertThat(realized.getResultType(), equalTo(Nullable.FLOATING));

    // now try with first arg as the nullable
    realize(inputType, parse("sum(value2, value)"));
    assertThat(realizationProblems, empty());
    assertThat(realized.evaluate(input), equalTo(5.2D));
    assertThat(realized.getResultType(), equalTo(Nullable.FLOATING));
  }

  @Test
  public void branchesCanReturnDifferentTypesOfNumbers() {
    // and they get coerced to floating
    build("test",
        "argument-types:",
        "  i: integer",
        "filter: i < 10",
        "  function: 10",
        "default: 20.0");

    Struct myInputType = Struct.of("value", Types.INTEGER);
    realize(myInputType, parse("test(value)"));

    assertThat(realizationProblems, empty());
    assertThat(realized.getResultType(), equalTo(Types.FLOATING));
    assertThat(realized.evaluate(Tuple.ofValues(myInputType, 9L)), equalTo(10.0D));
    assertThat(realized.evaluate(Tuple.ofValues(myInputType, 19L)), equalTo(20.0D));
  }

  @Test
  public void branchesCanReturnDifferentTypesOfNumbersFollowedByPostMath() {
    // and they get coerced to floating
    build("test",
        "argument-types:",
        "  i: integer",
        "filter: i < 10",
        "  function:",
        "    x: 10",
        "default:",
        "  function:",
        "    x: 20.0",
        "post: x * 3");

    Struct myInputType = Struct.of("value", Types.INTEGER);
    realize(myInputType, parse("test(value)"));

    assertThat(realizationProblems, empty());
    assertThat(realized.getResultType(), equalTo(Types.FLOATING));
    assertThat(realized.evaluate(Tuple.ofValues(myInputType, 9L)), equalTo(30.0D));
    assertThat(realized.evaluate(Tuple.ofValues(myInputType, 19L)), equalTo(60.0D));
  }

  @Test
  public void canAcceptAnything() {
    build("echo_anything",
        "argument-types:",
        "  arg: anything",
        "function: str(arg)"
    );
    assertThat(evaluate("echo_anything(10)", null), is("10"));
    assertThat(evaluate("echo_anything(1.5)", null), is("1.5"));
    assertThat(evaluate("echo_anything('ten')", null), is("ten"));
    assertThat(evaluate("echo_anything({foo: 'bar'})", null), is("{foo=bar}"));
  }

  @Test
  public void requiresCorrectTypesToRealize() {
    // GL336, till now classifier functions have been realized with whatever types where given
    // (unless function adaptation could improve this). This had meant that wrapping types may get
    // ignored. Now the function will not be realized unless the types match.
    IdentifiedFunction function = build("wrapped",
        "argument-types:",
        " construction: enum('wood', 'steel', 'concrete')",
        " break: set(text, 'breakfast', 'lunch', 'dinner')",
        " popularity: range(integer, 0, 10)",
        "default:",
        " function:",
        "   construction: construction",
        "   break: break",
        "   popularity: popularity");

    Struct wrappedType = Struct.of("a1", Types.INTEGER, "a2", Types.TEXT, "a3", Types.INTEGER);
    realize(wrappedType, parse("wrapped(a1, a2, a3)"));

    assertThat(realizationProblems, contains(
        hasAncestorProblem(is(ArgsProblems.mismatch(function.getArguments().get(0), Types.INTEGER)))
    ));
  }

  @Test
  public void requiresCorrectTypesInStructsToRealize() throws Exception {
    IdentifiedFunction function = build("wrapped",
        "argument-types:",
        "  arg:",
        "    type: enum('wood', 'steel', 'concrete')",
        "    value: floating",
        "function: arg.value * arg.value"
    );

    Struct myInputType = Struct.of("type", Types.INTEGER, "value", Types.INTEGER);
    realize(Struct.of("value", myInputType), parse("wrapped(value)"));

    assertThat(realizationProblems, contains(
        hasAncestorProblem(is(ArgsProblems.mismatch(function.getArguments().get(0), myInputType)))
    ));
  }

  @Test
  public void returnsMeaningfulErrorWhenOneFunctionCallsAnotherWithBadArgs() throws Exception {
    // example from GL227
    IdentifiedFunction sampleCurve = build("sample_curve",
        "argument-types:",
        "  x: floating",
        "  y: floating",
        "return-type: floating",
        "default:",
        "  function: (x ** 2) / ((x + 1) * 3.14)");

    // this function calls sample_curve with the arguments the wrong way around
    IdentifiedFunction func = build("func",
        "argument-types:",
        "  material: text",
        "return-type: floating",
        "default: sample_curve('foo', 'bar')");

    realize(Struct.EMPTY_STRUCT, parse("func('foo')"));

    assertThat(realizationProblems.get(0), allOf(
        hasAncestorProblem(is(ArgsProblems.mismatch(sampleCurve.getArguments().get(0), Types.TEXT))),
        hasAncestorProblem(is(ArgsProblems.mismatch(sampleCurve.getArguments().get(1), Types.TEXT)))
    ));
  }

  AST.FunctionDecl parse(String... lines) {
    return parser.parse(Arrays.asList(lines).stream().collect(Collectors.joining("\n")) + "\n");
  }

  IdentifiedFunction build(String id, String... lines) {
    ClassifierFunction built = new ClassifierFunction(parse(lines), project);
    project.getFunctionSet().remove(id);
    IdentifiedFunction identified = built.identified(id);
    project.getFunctionSet().add(identified);
    return identified;
  }

}
