/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.auth;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.engine.DefaultEngine;
import nz.org.riskscape.engine.ProjectTest;
import nz.org.riskscape.engine.auth.Secret;
import nz.org.riskscape.engine.auth.SecretBuilders;
import nz.org.riskscape.engine.ini.IniFileSecretBuilder;
import nz.org.riskscape.problem.ProblemSink;

/**
 * Tests how {@link IniFileSecretBuilder} builds the secrets from the defaults plugin.
 *
 * NB It would be nice to fold these tests in to their respective framework specific test cases, but that's difficult at
 * the moment with how the various units are factored (and located on the classpath).
 */
public class SecretBuilderTest extends ProjectTest {

  IniFileSecretBuilder builder;

  @Override
  protected void populateEngineWithDefaults(DefaultEngine newEngine) {
    super.populateEngineWithDefaults(newEngine);
    newEngine.getCollection(SecretBuilders.class).add(PlatformSecret.BUILDER);
    newEngine.getCollection(SecretBuilders.class).add(KoordinatesKeySecret.BUILDER);
    builder = new IniFileSecretBuilder(newEngine);
  }

  @Test
  public void canBuildKoordinatesSecret() {
    KoordinatesKeySecret secret = build("linz", """
                                                framework = koordinates
                                                hostname = foobar.com
                                                api-key = super secret
                                                """).isA(KoordinatesKeySecret.class).get();
    assertThat(secret.getId(), is("linz"));
    assertThat(secret.getHostname(), is("foobar.com"));
    assertThat(secret.getApiKey(), is("super secret"));
  }

  @Test
  public void canBuildKoordinatesSecretHostnameDefaultToID() {
    KoordinatesKeySecret secret = build("data.linz", """
                                                framework = koordinates
                                                api-key = super secret
                                                """).isA(KoordinatesKeySecret.class).get();
    assertThat(secret.getId(), is("data.linz"));
    assertThat(secret.getHostname(), is("data.linz"));
    assertThat(secret.getApiKey(), is("super secret"));
  }

  @Test
  public void canBuildPlatformSecret() {
    PlatformSecret secret = build("platform", """
                                                framework = platform
                                                hostname = foobar.com
                                                token = super secret
                                                """).isA(PlatformSecret.class).get();
    assertThat(secret.getId(), is("platform"));
    assertThat(secret.getHostname(), is("foobar.com"));
    assertThat(secret.getToken(), is("super secret"));
  }

  @Test
  public void canBuildPlatformSecretHostnameDefaultsRiskscapeNz() {
    PlatformSecret secret = build("platform", """
                                                framework = platform
                                                token = super secret
                                                """).isA(PlatformSecret.class).get();
    assertThat(secret.getId(), is("platform"));
    assertThat(secret.getHostname(), is("riskscape.nz"));
    assertThat(secret.getToken(), is("super secret"));
  }

  private Secret build(String id, String body) {
    loadConfig("secrets.ini", ProblemSink.DEVNULL,
        String.format("""
        [%s]
        %s
        """, id, body));

    builder.addSecret(id, config.getSection(id).get());

    return engine.getCollectionByClass(Secret.class)
        .get(id, engine.getProblemSink());
  }
}
