/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.auth;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.engine.auth.HttpSecretTest;
import nz.org.riskscape.engine.auth.SecretBuilder;

public class KoordinatesKeySecretTest extends HttpSecretTest<KoordinatesKeySecret> {

  public KoordinatesKeySecretTest() {
    params.put("id", "foo");
    params.put("hostname", "localhost");
    params.put("api-key", "shhhh!");
  }

  @Override
  public SecretBuilder getSecretBuilder() {
    return KoordinatesKeySecret.BUILDER;
  }

  @Override
  protected String getMatchingPath() {
    return "/services/";
  }

  @Override
  protected String getSecretValue() {
    return secret.getApiKey();
  }

  @Test
  public void secretOnlyAppliesWhenServicesIsLeadingPathElement() {
    // defaults
    buildSecret();

    // no services in path
    assertFalse(applySecret("https://localhost").isSecretApplied());
    assertFalse(applySecret("https://localhost/foo/bar").isSecretApplied());

    // services, but in the wrong place (needs to be first)
    assertFalse(applySecret("https://localhost/foo/services/bar").isSecretApplied());

    // doesn't just match on starting with /services
    assertFalse(applySecret("https://localhost/services-foo/bar").isSecretApplied());

    // services is now in the correct place, key is inserted
    assertThat(
        applySecret("https://localhost/services/foo/bar?a=b"),
        allOf(
            hasProperty("path", equalTo("/services;key=shhhh!/foo/bar")),
            hasProperty("headers", empty())
        )
    );
  }

  @Test
  public void aKeyInThePathIsReplaced() throws Exception {
    buildSecret();

    // TODO Might be nice to warn when we see this and give the user a replacement instead
    assertThat(
        applySecret("https://localhost/services;key=deadbeef/foo"),
        allOf(
            hasProperty("path", equalTo("/services;key=shhhh!/foo")),
            hasProperty("headers", empty())
        )
    );
  }

  @Test
  public void aMangledKeyInThePathIsReplaced() throws Exception {
    buildSecret();

    assertThat(
        applySecret("https://localhost/services;key=/foo"),
        hasProperty("path", equalTo("/services;key=shhhh!/foo"))
    );

    assertThat(
        applySecret("https://localhost/services;key/foo"),
        hasProperty("path", equalTo("/services;key=shhhh!/foo"))
    );

    assertThat(
        applySecret("https://localhost/services;/foo"),
        hasProperty("path", equalTo("/services;key=shhhh!/foo"))
    );
  }

  @Test
  public void queryParametersAreLeftAlone() throws Exception {
    buildSecret();

    assertThat(
        applySecret("https://localhost/services/foo?crs=wgs84").getURI().toString(),
        equalTo("https://localhost/services;key=shhhh!/foo?crs=wgs84")
    );
  }
}
