/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.util.HashMap;
import java.util.Map;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.defaults.function.LookupFunction;
import nz.org.riskscape.defaults.lookup.MapLookupTable;
import nz.org.riskscape.engine.Matchers;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.lookup.LookupTable;
import nz.org.riskscape.engine.rl.BaseExpressionRealizerTest;
import nz.org.riskscape.engine.types.LookupTableType;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinSet;

public class LookupFunctionTest extends BaseExpressionRealizerTest {

  Map<String, Long> hashMap = new HashMap<>();
  LookupTable table = new MapLookupTable(hashMap, Types.TEXT, Types.INTEGER);
  Tuple input = Tuple.ofValues(Struct.of("table", table.getLookupTableType()), table);

  @Before
  public void setup() {
    project.getFunctionSet().add(new LookupFunction().builtin("lookup", IdentifiedFunction.Category.MISC));
  }

  @Test
  public void returnsTheValueTypeOfTheTableButNullable() {

    hashMap.put("bar", 420L);

    evaluate("lookup(table, 'foo')", input);
    assertEquals(Nullable.INTEGER, realized.getResultType());
    assertNull(evaluated);

    evaluate("lookup(table, 'bar')", input);
    assertEquals(420L, evaluated);
  }

  @Test
  public void keyOnlyNeedsToBeCovariantToTableKeyType() throws Exception {
    input = Tuple.ofValues(Struct.of(
        "table", table.getLookupTableType(),
        "some_key", new WithinSet(Types.TEXT, "foo")
    ), table, "foo");

    hashMap.put("bar", 420L);

    evaluate("lookup(table, some_key)", input);
    assertEquals(Nullable.INTEGER, realized.getResultType());
    assertNull(evaluated);

    evaluate("lookup(table, 'bar')", input);
    assertEquals(420L, evaluated);
  }

  @Test
  public void failsIfFirstArgNotALookupTable() throws Exception {
    evaluate("lookup('foo', 'bar')", Tuple.EMPTY_TUPLE);

    assertThat(
        realizationProblems,
        contains(
            Matchers.hasAncestorProblem(equalTo(TypeProblems.get().mismatch(expr, LookupTableType.WILD, Types.TEXT)))
        )
    );
  }

  @Test
  public void failsIfGivenKeyNotCovariantWithTablesKeyType() throws Exception {
    evaluate("lookup(table, 2)", input);

    assertThat(
        realizationProblems,
        contains(
            Matchers.hasAncestorProblem(equalTo(TypeProblems.get().mismatch(expr, Types.TEXT, Types.INTEGER)))
        )
    );
  }
}
