/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.output;

import java.io.File;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.PrintWriter;
import java.net.URI;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.MessageDigest;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import com.google.common.io.Files;

import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.cli.ExitException;
import nz.org.riskscape.engine.pipeline.Manifest;
import nz.org.riskscape.engine.pipeline.RealizedPipeline;
import nz.org.riskscape.engine.pipeline.sink.SaveSink;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.resource.CreateException;
import nz.org.riskscape.engine.resource.CreateHandle;
import nz.org.riskscape.engine.resource.CreateHandle.Callback;
import nz.org.riskscape.engine.resource.CreateRequest;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.types.Geom;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.util.StatsWriter;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * A {@link PipelineOutputStore} that writes results to files on the file system.
 *
 * If the user specifies an output base URI then that is used as the directory in which results are saved.
 * If not provided then the output base directory is PROJECT_OUTPUT_DIRECTORY/PIPELINE_ID/TIMESTAMP.
 */
@Slf4j
public class FileSystemPipelineOutputStore implements PipelineOutputStore {

  /**
   * Callback to write the stats to file when the FileSystemPipelineContainer is closed
   */
  private static final Consumer<FileSystemPipelineContainer> WRITE_STATS = container -> {
    StatsWriter writer = new StatsWriter(container.getPipeline().getContext().getMetricRegistry());
    container.writeFile(StatsWriter.FILENAME, StatsWriter.CONTENT_TYPE, os -> writer.writeStats(os));
  };

  @Getter
  private final String id = "filesystem";
  private final Supplier<LocalDateTime> currentTime;

  private final List<Consumer<FileSystemPipelineContainer>> completionHooks = new ArrayList<>();

  class FileSystemPipelineContainer extends BasePipelineOutputContainer {

    final CreateRequest outputBase;
    final Path baseDir;

    FileSystemPipelineContainer(CreateRequest outputBase, RealizedPipeline pipeline,
        PipelineOutputOptions options) {
      super(FileSystemPipelineOutputStore.this, pipeline, options, FileSystemPipelineOutputStore.this.currentTime);
      this.outputBase = outputBase;
      this.baseDir = Paths.get(outputBase.getContainer());
    }

    @Override
    public URI getStoredAt() {
      return outputBase.getContainer();
    }

    /**
     * Helper to simplify creating a new file handle and storing the data. This is useful
     * for writing small amounts of metadata that can be saved all in one go, e.g. stats.txt
     */
    public void writeFile(String name, String contentType, Callback writeCallback) throws CreateException {
      getResourceFactory().create(outputBase, name, contentType).store(writeCallback);
    }

    @Override
    protected ResultOrProblems<SaveSink> createSink(SinkParameters params) {
      return ProblemException.catching(() -> {
        Format format = params.getFormat().orElse(pickDefaultFormat(params.getType()));
        String extension = Files.getFileExtension(params.getName());

        // shapefile format demands that it ends with .shp, or it won't write it, and it doesn't hurt to force the
        // extension for the other formats, too
        String extendedName;
        if (!extension.equals(format.getExtension())) {
          extendedName = params.getName() + "." + format.getExtension();
        } else {
          extendedName = params.getName();
        }

        CreateHandle handle = getResourceFactory()
            .create(outputBase, extendedName, format.getMediaType());

        WriterConstructor writerConstructor = format.getWriterConstructor().orElseThrow(()
            -> new ProblemException(GeneralProblems.get().operationNotSupported("output", format.getClass())));

        return writerConstructor
            .newWriter(getExecutionContext(), params.getType(), handle, params.getFormatOptions())
            .map(writer -> new SaveSink(writer))
            .drainWarnings(p -> getEngine().getProblemSink().accept(p))
            .getOrThrow(Problems.foundWith(format));
      });
    }

    Format pickDefaultFormat(Struct inputType) throws ProblemException {
      if (options.getFormat().isPresent()) {
        // NB don't think this   should ever happen, as we check options before calling this method, but code has a
        // habit of changing, so let's leave this small nod to robustness
        return options.getFormat().get();
      } else if (containsGeometry(inputType)) {
        // we expect there to be a format named 'shapefile'
        return getEngine().getFormats().getOr("shapefile").getOrThrow();
      } else {
        // we expect there to be a format named 'csv'
        return getEngine().getFormats().getOr("csv").getOrThrow();
      }
    }

    private boolean containsGeometry(Struct struct) {
      for (StructMember member : struct.getMembers()) {
        Type memberType = Nullable.strip(member.getType());
        if (memberType.find(Geom.class).isPresent()) {
          return true;
        }
        Optional<Struct> structMember = memberType.find(Struct.class);
        if (structMember.isPresent()) {
          if (containsGeometry(structMember.get())) {
            return true;
          }
        }
      }

      return false;
    }

    @Override
    public void close() {
      // set the finish time
      manifest.finishTime = currentTime.get();

      moveRegisteredFileOutputs();

      // add the output files to manifest
      addOutputInfoToManifest(manifest);

      // add the pipeline source if it is known
      Optional<String> pipelineSource = getPipelineSource();
      if (pipelineSource.isPresent()) {
        CreateHandle dslHandle = getResourceFactory().create(
            outputBase, "pipeline.txt", "text/plain");
        try (PrintWriter writer = new PrintWriter(new OutputStreamWriter(dslHandle.getOutputStream()))) {
          writer.write(pipelineSource.get());
        }

        manifest.add(createOutputInfo("pipeline-source", dslHandle.store()));
      }

      // write the manifest
      writeFile(Manifest.MANIFEST_FILE, "text/plain", os -> manifest.write(os));

      FileSystemPipelineOutputStore.this.completionHooks.forEach(action -> action.accept(this));
    }

    private void moveRegisteredFileOutputs() {
      for (Path path: fileOutputs) {
        try {
          String fileName = path.getFileName().toString();
          if (path.startsWith(baseDir)) {
            // The user has written the output file directly to the output directory. No need to move it.

            // Keep the displayed URI looking the same as the other outputs
            // toFile().toURI() -> file:/ vs .toUri() -> file:///
            URI uri = path.toFile().toURI();
            movedFileOutputs.put(fileName, uri);
          } else {
            CreateHandle handle = getResourceFactory().create(outputBase, fileName, null);

            java.nio.file.Files.copy(path, handle.getOutputStream());
            java.nio.file.Files.delete(path);
            movedFileOutputs.put(fileName, handle.store());
          }
        } catch (IOException e) {
          // I wonder if we could do better here...  We could log a warning and record the old location instead?
          throw new RiskscapeException(Problems.caught(e));
        }
      }
    }

    /**
     * Adds checksum/byte info for the output files produced by the pipeline to the manifest
     */
    private void addOutputInfoToManifest(Manifest manifest) {
      for (Entry<String, SaveSink> entry : sinks.entrySet()) {
        SaveSink instance = entry.getValue();
        manifest.add(createOutputInfo(entry.getKey(), instance.getStoredAt()));
      }

      for (Entry<String, URI> entry : movedFileOutputs.entrySet()) {
        manifest.add(createOutputInfo(entry.getKey(), entry.getValue()));
      }
    }

    private Manifest.OutputInfo createOutputInfo(String codename, URI outputTo) {

      String size;
      String digest;
      String name;

      if (outputTo.getScheme().equals("file")) {
        File file = new File(outputTo.getPath());
        size = Long.toString(file.length());

        if (options.isChecksum()) {
          if (Files.getFileExtension(file.getName()).equalsIgnoreCase("shp")) {
            log.warn("Incomplete checksum for output step '{}' - Checksumming for shp files *only* checks .shp "
                + "file - not .dbx, etc - ", codename);
          }
          digest = createDigest(outputTo);
        } else {
          digest = null;
        }
        name = file.getName();
      } else {
        // NB in the future, we will/might want to update the Resource class to include some of this metadata
        log.warn("{} is not a file, manifest will be incomplete");
        size = "unknown";
        digest = null;
        name = outputTo.toString();
      }

      return new Manifest.OutputInfo(
          name,
          size,
          digest
      );
    }

    private String createDigest(URI storedAt) {
      MessageDigest digest = Manifest.getDigestInstance();

      Resource resource = getResourceFactory().load(storedAt);
      java.io.InputStream stream = resource.getContentStream();
      try {
        byte[] chunk = new byte[4096];
        int read = 0;
        while ((read = stream.read(chunk)) != -1) {
          digest.update(chunk, 0, read);
        }

        return Manifest.checksum(digest);
      } catch (IOException e) {
        tryAndClose(stream);
        throw new ExitException(1, e, "Failed to create digest");
      } finally {
        tryAndClose(stream);
      }
    }

    /**
     * Attempt to close it, warn if we can't
     */
    private void tryAndClose(AutoCloseable closeable) {
      try {
        closeable.close();
      } catch (Exception e) {
        log.warn("Failed to close manifest content input stream", e);
      }
    }

  }

  public FileSystemPipelineOutputStore() {
    this(() -> LocalDateTime.now());
  }

  /**
   * Test constructor allowing control of the current time.
   */
  protected FileSystemPipelineOutputStore(Supplier<LocalDateTime> currentTime) {
    this.currentTime = currentTime;
    onCompletion(WRITE_STATS);
  }

  @Override
  public int isApplicable(URI outputLocation) {
    return "file".equals(outputLocation.getScheme()) ? PRIORITY_DEFAULT : PRIORITY_NA;
  }

  @Override
  public ResultOrProblems<PipelineOutputContainer> create(URI outputLocation, RealizedPipeline pipeline,
      PipelineOutputOptions options) {
    CreateRequest createRequest = new CreateRequest(outputLocation, null, null, options.isReplace());
    return ResultOrProblems.of(new FileSystemPipelineContainer(createRequest, pipeline, options));
  }

  /**
   * Register an action to perform once a FileSystemPipelineContainer has
   * completed saving all the output data.
   */
  public void onCompletion(Consumer<FileSystemPipelineContainer> action) {
    completionHooks.add(action);
  }
}
