/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.function.geometry;

import java.util.List;
import java.util.function.Function;

import org.locationtech.jts.geom.Geometry;

import com.google.common.collect.Lists;

import lombok.Getter;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.CoverageType;
import nz.org.riskscape.engine.types.Geom;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class SampleCoverage extends BaseSamplingFunction {

  @Getter
  private final Type returnType = RSList.create(Struct.of("geometry", Types.GEOMETRY, "sampled", Types.ANYTHING));

  @Override
  protected ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
    Geom geomType, CoverageType coverageType) {

    Struct struct = Struct.of("geometry", Types.GEOMETRY, "sampled", coverageType.getMemberType());
    Type returnList = RSList.create(struct);

    return ResultOrProblems.of(new RiskscapeFunction() {
      @Getter
      private final ArgumentList arguments = ArgumentList.create("geometry", geomType, "coverage", coverageType);

      @Getter
      private final List<Type> argumentTypes = arguments.getArgumentTypes();

      @Getter
      private final Type returnType = returnList;

      @Override
      public Object call(List<Object> args) {
        Geometry geom = (Geometry) args.get(0);
        TypedCoverage coverage = (TypedCoverage) args.get(1);

        Function<Geometry, List<Pair<Geometry, Object>>> op = coverage.getEvaluateIntersectionOp().orElse(null);

        if (op == null) {
          throw new RiskscapeException(
              GeneralProblems.get().operationNotSupported("intersection", coverage.getClass()));
        }

        return Lists.transform(op.apply(geom), pair -> Tuple.ofValues(struct, pair.getLeft(), pair.getRight()));
      }
    });
  }

}
