/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.defaults.data;

import java.io.IOException;
import java.net.MalformedURLException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;

import nz.org.riskscape.problem.Problems;
import org.geotools.api.data.DataStore;
import org.geotools.api.referencing.crs.CoordinateReferenceSystem;
import org.geotools.geopkg.GeoPkgDataStoreFactory;

import com.google.common.collect.ImmutableMap;

import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.bind.JavaParameterSet;
import nz.org.riskscape.engine.data.relation.FeatureSourceBookmarkResolver;
import nz.org.riskscape.engine.data.relation.RelationBookmarkParams;
import nz.org.riskscape.engine.projection.ForceSridProjection;
import nz.org.riskscape.engine.relation.FeatureSourceRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.problem.ResultOrProblems;

public class GeoPackageRelationResolver extends FeatureSourceBookmarkResolver<DataStore, RelationBookmarkParams> {

  public static final Map<String, String> EXTENSIONS_TO_FORMATS
      // note that the format is geopackage as we expect geopackage features (relation) to be the primary use-case.
      // This means that something like geopackage-coverage may be required to support rasters from geopackages
      = ImmutableMap.of("gpkg", "geopackage");

  private final GeoPkgDataStoreFactory datastoreFactory = new RsGpkgDataStoreFactory();

  public GeoPackageRelationResolver(Engine engine) {
    super(engine);
  }

  @Override
  protected DataStore createDataStore(RelationBookmarkParams bookmark)
      throws MalformedURLException, IOException {
    Path geopkgPath = getBookmarkedPath(bookmark);

    Map<String, Object> params = new HashMap<>();
    params.put("dbtype", "geopkg");
    params.put("database", geopkgPath.toString());
    params.put("read_only", true);
    return datastoreFactory.createDataStore(params);
  }

  @Override
  protected Problems checkDataValid(RelationBookmarkParams params, DataStore dataStore) {
    try {
      // this is the first thing that tries to establish an SQL connection, so that's
      // usually where things fall over if the file is bad
      dataStore.getTypeNames();
      return null;
    } catch (IOException | RuntimeException ex) {
      return Problems.caught(ex);
    }
  }

  @Override
  protected Map<String, String> getExtensionsToFormats() {
    return EXTENSIONS_TO_FORMATS;
  }

  @Override
  protected JavaParameterSet<RelationBookmarkParams> buildParameterSet() {
    RelationBookmarkParams params = new RelationBookmarkParams();
    // the geopackage format specifies an XY (long/lat) axis order. So we make this the default.
    // Otherwise the user would need to set the crs-name and crs-longitude-first
    // correctly or risk their geometries been completely wrong.
    params.crsLongitudeFirst = true;
    return JavaParameterSet.fromBindingInstance(RelationBookmarkParams.class, params);
  }

  @Override
  protected ResultOrProblems<Relation> wrapRelation(RelationBookmarkParams params, DataStore dataStore,
      FeatureSourceRelation relation, CoordinateReferenceSystem crs) {
    // The GeoPackage data store ignores the query hints setting Hints.JTS_GEOMETRY_FACTORY. Instead
    // it uses it's own geometry factory which will set the geometry SRID to the EPSG value, eg 2193
    // for NZTM.
    // So we use the ForceSridProjection to update the SRID and geometry factory in the geometries.
    return relation.project(new ForceSridProjection(crs, params.getProject().getSridSet()));
  }

}
