/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.defaults.data;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.net.URI;
import java.net.URLDecoder;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.util.Optional;

import com.google.common.base.Strings;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.RiskscapeIOException;
import nz.org.riskscape.engine.resource.BaseResource;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.resource.ResourceLoader;
import nz.org.riskscape.engine.resource.ResourceLoadingException;
import nz.org.riskscape.problem.Problems;

/**
 * Supports loading data embedded in URIs (RFC-2397).  See ByteArrayCreateHandle for methods that can generate these
 * URIs (useful for tests)
 */
@RequiredArgsConstructor
public class DataURILoader implements ResourceLoader {

  private final Engine engine;

  public static final Charset DEFAULT_ENCODING = StandardCharsets.US_ASCII;

  @Override
  public String getId() {
    return "data";
  }

  @Override
  public String getDescription() {
    return "Supports data embedded in URIs (RFC-2397)";
  }

  @Override
  public boolean canLoad(URI target) {
    return getId().equals(target.getScheme());
  }

  @Override
  public Resource load(URI target) throws ResourceLoadingException {

    String specificPart = target.getRawSchemeSpecificPart();
    String[] mediaTypeAndData = specificPart.split(",");

    if (mediaTypeAndData.length != 2) {
      throw new ResourceLoadingException(target, "");
    }

    String mediaTypePart = mediaTypeAndData[0].trim();
    boolean base64 = false;

    final String base64Suffix = ";base64";

    if (mediaTypePart.endsWith(base64Suffix)) {
      base64 = true;
      mediaTypePart = mediaTypePart.substring(mediaTypePart.length() - base64Suffix.length());
    }

    // find charset - we need that to un-url encode
    Charset charset = DEFAULT_ENCODING;
    String[] mediaTypeSplit = mediaTypePart.split(";");
    for (String bit : mediaTypeSplit) {
      String[] keyValue = bit.split("=");
      if (keyValue[0].equals("charset")) {
        charset = Charset.forName(keyValue[1]);
      }
    }

    String mediaType;
    if (Strings.isNullOrEmpty(mediaTypePart)) {
      mediaType = "text/plain";
    } else {
      mediaType = mediaTypePart;
    }

    byte[] bytes;
    String dataPart = mediaTypeAndData[1];
    try {
      if (base64) {
        // as far as I can tell, this should always be us-ascii - these are opaque bytes and it is up
        // to the consumer of the resource to respect the media type and extract the bytes correctly
        bytes = Base64.getUrlDecoder().decode(dataPart.getBytes(DEFAULT_ENCODING));
      } else {
        bytes = URLDecoder.decode(dataPart, charset).getBytes();
      }
    } catch (IllegalArgumentException e) {
      throw new ResourceLoadingException(target, Problems.caught(e));
    }

    return new BaseResource(target) {

      @Override
      public InputStream getContentStream() throws RiskscapeIOException {
        return new ByteArrayInputStream(bytes);
      }

      @Override
      public Optional<String> getMediaType() {
        return Optional.of(mediaType);
      }

    };
  }

}
