/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.interp;

import java.util.Arrays;
import java.util.List;
import java.util.function.Consumer;

import com.google.common.primitives.Doubles;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.rl.ScopedLambdaExpression;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeVisitor;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.Expression;


/**
 * The riskscape type representing a continuous function resulting from the {@link CreateContinuousFunction} function.
 *
 * Has the logic and static state (x values and realized expressions) required for computing y values for a
 * {@link ContinuousFunction} object.  You might expect that this logic and state would on the {@link
 * ContinuousFunction} object instead, but to keep that code clean we would want to have all the type-static state to
 * live on the ContinuousFunction which would blow out its memory footprint.  Otherwise we'd need to structure the
 * ContinuousFunction methods so that they took a type as their first argument, like some sort of mad, twisted python
 * program.
 *
 * Marked final to (possibly) help the JVM to optimize this code.  Can be removed if a future refactor deems it awkward.
 */
@EqualsAndHashCode(callSuper = true, of = {"xValues"})
final class LinearContinuousFunctionType extends StackableContinuousFunctionType<LinearContinuousFunction> {

  @RequiredArgsConstructor
  public static class Builder {
    public final RealizationContext context;

    public RealizedExpression yExpression;
    public double[] xValues;
    public Type xType;
    public boolean logScale = false;

    public ResultOrProblems<LinearContinuousFunctionType> build(Consumer<Builder> setupVisitor) {
      setupVisitor.accept(this);

      return ProblemException.catching(() -> {

        Type yType = yExpression.getResultType();

        //                                                scaling x axis with log            unscaled (linear x)
        String slopeExprString      = logScale ? "(y1 - y0) / (log(x1) - log(x0))" : "(y1 - y0) / (x1 - x0)";
        String yInterceptExprString = logScale ? "y1 - (m * log(x1))"              : "y1 - (m * x1)";
        String solveForYExprString  = logScale ? "(m * log(x)) + c"                : "(m * x) + c";

        Struct slopeInputType = Struct.of(
            "x0", xType,
            "x1", xType,
            "y0", yType, "y1", yType
        );

        RealizedExpression slopeExpression = realize(slopeInputType, slopeExprString);
        Struct yInterceptInputType = Struct.of(
            "y1", yType,
            "m", slopeExpression.getResultType(), "x1", Types.FLOATING
        );

        RealizedExpression yInterceptExpression = realize(yInterceptInputType, yInterceptExprString);
        Struct solveForYExpressionInputType = Struct.of(
            "m", slopeExpression.getResultType(),
            "x", Types.FLOATING,
            "c", yInterceptExpression.getResultType()
        );

        RealizedExpression solveForYExpression = realize(solveForYExpressionInputType, solveForYExprString);

        return new LinearContinuousFunctionType(xValues, yExpression,
            slopeInputType, slopeExpression,
            yInterceptInputType, yInterceptExpression,
            solveForYExpressionInputType, solveForYExpression
        );
      });
    }

    public RealizedExpression realize(Struct inputType, String expr) throws ProblemException {
      Expression parsed = ExpressionParser.parseString(expr);
      return context.getExpressionRealizer().realize(inputType, parsed).getOrThrow();
    }
  }

  public static final LinearContinuousFunctionType ANY = new LinearContinuousFunctionType();

  public static final Object VISITOR_META_CVALUE = "cvalue";
  public static final Object VISITOR_META_MVALUE = "mvalue";
  public static final Object VISITOR_META_YVALUE = "yvalue";

  @Getter
  private final double[] xValues;

  // expression for computing the slope of a line formed between two points, computed from a pair of neighbouring x-y
  // coordinates
  private final Struct slopeInputType;
  private final RealizedExpression slopeExpression;

  // expression for computing y-intercepts from the slope (m) and a known x and y
  private final Struct yInterceptInputType;
  private final RealizedExpression yInterceptExpression;

  // expression for computing an interpolated y value from a slope and a y intercept
  private final Struct solveForYExpressionInputType;
  private final RealizedExpression solveForYExpression;

  LinearContinuousFunctionType(double[] xValues, RealizedExpression yExpression,
      Struct slopeInputType, RealizedExpression slopeExpression,
      Struct yInterceptInputType, RealizedExpression yInterceptExpression,
      Struct solveForYExpressionInputType, RealizedExpression solveForYExpression) {
    super(ONE_DIMENSIONAL, yExpression, solveForYExpression.getResultType(), false);
    this.xValues = xValues;
    this.slopeInputType = slopeInputType;
    this.slopeExpression = slopeExpression;
    this.yInterceptInputType = yInterceptInputType;
    this.yInterceptExpression = yInterceptExpression;
    this.solveForYExpressionInputType = solveForYExpressionInputType;
    this.solveForYExpression = solveForYExpression;
  }

  private LinearContinuousFunctionType() {
    super(ONE_DIMENSIONAL, null, Types.ANYTHING, false);
    this.xValues = new double[0];
    slopeExpression = null;
    slopeInputType = null;
    solveForYExpression = null;
    solveForYExpressionInputType = null;
    yInterceptExpression = null;
    yInterceptInputType= null;
  }

  @Override
  public String toString() {
    return String.format("ContinuousCurve(xvalues=%s, returnType=%s)", Doubles.asList(xValues), returnType);
  }

  public Type getCValueType() {
    return yInterceptExpression.getResultType();
  }

  public Type getMValueType() {
    return slopeExpression.getResultType();
  }

  @Override
  public Object applyTo(Object func, double... dimensionValues) {
    LinearContinuousFunction function = (LinearContinuousFunction)func;
    double xValue = dimensionValues[0];

    int size = xValues.length;
    // TODO here we assume a constant value based on the upper and lower bounds of the function, i.e. a horizontal line
    // from before x0 and after xn
    if (xValue < xValues[0]) {
      return solveForY(function, xValues[0], 0);
    } else if (xValue >= xValues[size - 1]) {
      return solveForY(function, xValues[size - 1], size - 2);
    }

    for (int i = 0; i < size - 1; i++) {
      double x0 = xValues[i];
      double x1 = xValues[i + 1];

      if (xValue >= x0 && xValue < x1) {

        return solveForY(function, xValue, i);
      }
    }

    throw new AssertionError("should not be possible");
  }

  private Object solveForY(LinearContinuousFunction function, double xValue, int index) {
    Object slope = getOrComputeSlope(function, index);
    Object yIntercept = getOrComputeYIntercept(function, index);

    return solveForY(slope, yIntercept, xValue);
  }

  private Object solveForY(Object slope, Object yIntercept, double xValue) {
    Tuple inputType = Tuple.ofValues(solveForYExpressionInputType, slope, xValue, yIntercept);
    return solveForYExpression.evaluate(inputType);
  }

  private Object getOrComputeYIntercept(LinearContinuousFunction function, int i) {
    if (function.cValues[i] == null) {
      Object m = function.mValues[i]; // slope
      Object y1 = getValue(function, i);
      double x1 = xValues[i];
      Tuple tuple = Tuple.ofValues(yInterceptInputType, y1, m, x1);
      function.cValues[i] = yInterceptExpression.evaluate(tuple);
    }
    return function.cValues[i];
  }

  private Object getOrComputeSlope(LinearContinuousFunction function, int i) {
    Object mValue = function.mValues[i];
    if (mValue == null) {
      Object y0 = getOrComputeValue(function, i);
      Object y1 = getOrComputeValue(function, i + 1);

      double x0 = xValues[i];
      double x1 = xValues[i + 1];

      Tuple tuple = Tuple.ofValues(slopeInputType, x0, x1, y0, y1);
      mValue = slopeExpression.evaluate(tuple);
      function.mValues[i] = mValue;
    }
    return mValue;
  }

  @Override
  protected Object getOrComputeValue(LinearContinuousFunction function, int i) {
    Object currentValue = getValue(function, i);

    if (currentValue == null) {
      // NB we could lock here instead?
      Tuple inputTuple = function.lambda.buildCallingScope((Struct) valueExpression.getInputType(), xValues[i]);
      inputTuple.set(inputTuple.size() - 1, xValues[i]);
      currentValue = valueExpression.evaluate(inputTuple);
      setValue(function, i, currentValue);
    }

    return currentValue;
  }

  @Override
  LinearContinuousFunction newFunction(ScopedLambdaExpression lambda) {
    return new LinearContinuousFunction(lambda, this);
  }

  @Override
  public int getSize() {
    return xValues.length;
  }

  @Override
  public <T, U> U visit(TypeVisitor<T, U> tv, T data) {
    List<Pair<Type, ?>> children = Arrays.asList(
        Pair.of(getCValueType(), VISITOR_META_CVALUE),
        Pair.of(getMValueType(), VISITOR_META_MVALUE),
        Pair.of(getValueType(), VISITOR_META_YVALUE)
    );
    return tv.compoundType(this, children, data);
  }
}
