/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.interp;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.CoercionException;
import nz.org.riskscape.engine.types.SimpleType;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeVisitor;
import nz.org.riskscape.engine.types.Types;

/**
 * The base riskscape type representing a continuous function resulting from the
 * {@link CreateContinuousFunction} function.
 *
 * Note, this is really a base class and implementations should implement {@link #applyTo(ContinuousFunction, double) }
 * and {@link #visit(TypeVisitor, Object) }
 *
 * # A Future Refactor...
 *
 * The ContinuousFunction and ContinuousFunctionType are quite confusing in that the type is much more than a type and
 * the function is not actually a function at all.  They are a worst of both worlds mix of structural and OO programming
 * (although a recent change has hopefully improved this a bit).
 *
 * I think the 'future' should consider the following changes:
 *
 *  - Drop the specific types, and instead increase the amount of metadata on the base type
 *  - Add an "implementation" object that does what the current types are doing
 *  - Consider dropping ContinuousFunction and instead use Tuples to store the state the functions require to operate
 *  on.
 *
 */
@RequiredArgsConstructor
@EqualsAndHashCode(callSuper = false, of = {"dimensions", "valueExpression", "returnType"})
class ContinuousFunctionType implements Type {

  public static final int ONE_DIMENSIONAL = 1;
  public static final int TWO_DIMENSIONAL = 2;

  public static final ContinuousFunctionType ANY = new ContinuousFunctionType(ONE_DIMENSIONAL, null, Types.ANYTHING);

  @Getter
  protected final int dimensions;

  /**
   * An expression to evaluate to plot the points of this curve type, in maths talk we can call this the interpolant.
   * The expression will be evaluated with {@link ContinuousFunction#lamba} and values for each of the curves
   * dimensions.
   */
  @Getter
  protected final RealizedExpression valueExpression;

  /**
   * The return type. Based on value expressions return type be should take account of interpolation.
   */
  @Getter
  protected final Type returnType;

  protected final Struct asStruct = Struct.of(SimpleType.DEFAULT_MEMBER_NAME, this);

  protected Type getValueType() {
    return valueExpression.getResultType();
  }

  @Override
  public Struct asStruct() {
    return asStruct;
  }

  @Override
  public Object coerce(Object value) throws CoercionException {
    throw new UnsupportedOperationException();
  }

  @Override
  public int estimateSize(Object entry) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Object fromBytes(DataInputStream in) throws IOException {
    throw new UnsupportedOperationException();
  }

  @Override
  public void toBytes(DataOutputStream os, Object toWrite) throws IOException {
    throw new UnsupportedOperationException();
  }

  /**
   * Compute an interpolated y for a given x
   * @param function the function to interpolate a y value for
   * @param dimensionValues a value for each of the dimensions this function accepts
   * @return a value corresponding to {@link #getYValueType()}
   */
  public Object applyTo(Object function, double... dimensionValues) {
    throw new UnsupportedOperationException();
  }

  @Override
  public <T, U> U visit(TypeVisitor<T, U> tv, T data) {
    throw new UnsupportedOperationException();
  }

  @Override
  public Class<?> internalType() {
    return ContinuousFunction.class;
  }

  @Override
  public <T extends Type> boolean isA(Class<T> type) {
    return type.isAssignableFrom(getClass());
  }

  @Override
  public String toString() {
    // this toString should only be used in function info type output.
    return String.format("ContinuousCurve(returnType=%s)", returnType);
  }

}
