/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.interp;


import java.util.Comparator;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.BinaryOperator;

import nz.org.riskscape.defaults.interp.BilinearContinuousFunctionType.XY;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.rl.ScopedLambdaExpression;
import nz.org.riskscape.engine.types.Struct;

/**
 * Marked final to (possibly) help the JVM to optimize this code.  Can be removed if a future refactor deems it awkward.
 */
final class BilinearContinuousFunction extends StackableContinuousFunction {

  // We store the maximal X/Y values for a zero loss. Initially we make these negative infinity which
  // is fine because we're not going to get real X/Y values that small and means we don't need to deal with
  // any null values.
  // Note that we could further reduce the memory footprint of each function by using a VarHandle and use
  // compare-and-set in the same way that AtomicInteger implements it.  This would save four extra object allocations
  // per function (one for the atomic ref, one for the Double wrapper, and then double it for each loss)
  private final AtomicReference<Double> zeroLossX = new AtomicReference<>(Double.NEGATIVE_INFINITY);
  private final AtomicReference<Double> zeroLossY = new AtomicReference<>(Double.NEGATIVE_INFINITY);

  private final BinaryOperator<Double> maxValueAccumulator = BinaryOperator.maxBy(Comparator.comparingDouble(x -> x));

  BilinearContinuousFunction(ScopedLambdaExpression lambda, StackableContinuousFunctionType<?> type) {
    super(lambda, type);
  }

  public Object evaluate(RealizedExpression realized, double x, double y) {
    Tuple scope = lambda.buildCallingScope((Struct) realized.getInputType(), x, y);
    return realized.evaluate(scope);
  }

  /**
   * Check if a zero loss can be returned. A zero loss can be returned if the function has already encountered
   * a zero loss with higher (or equal) x and y values.
   */
  public boolean useZeroLoss(XY given) {
    return given.x <= zeroLossX.get()
        && given.y <= zeroLossY.get();
  }

  /**
   * Mark a zero loss at the given position
   */
  public void addZeroLoss(XY given) {
    // we expand the zeroLoss X/Y to include the new given value.
    zeroLossX.accumulateAndGet(given.x, maxValueAccumulator);
    zeroLossY.accumulateAndGet(given.y, maxValueAccumulator);
  }
}
