/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.interp;

import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;

import lombok.Getter;
import lombok.RequiredArgsConstructor;

import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

/**
 * RiskscapeFunction for computing a y value from the result of the {@link CreateContinuousFunction} function.
 * {@link CreateContinuousFunction} has more details.
 */
public class ApplyContinuousFunction implements RealizableFunction {

  @RequiredArgsConstructor
  private static class FunctionImpl implements RiskscapeFunction {

    private final ContinuousFunctionType curveType;

    @Getter
    private final List<Type> argumentTypes;

    @Getter
    private final Type returnType;

    @Override
    public Object call(List<Object> args) {
      Object function = args.get(0);
      if (function == null) {
        return null;
      }

      double[] values = new double[curveType.dimensions];
      for (int i = 0; i < curveType.dimensions; i++) {
        values[i] = ((Number) args.get(i + 1)).doubleValue();
      }

      return curveType.applyTo(function, values);
    }
  }

  private final ArgumentList arguments = ArgumentList.create(
      "function", ContinuousFunctionType.ANY,
      "x-value", Types.FLOATING,
      "y-value", Nullable.FLOATING
  );

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> argumentTypes) {

    return ProblemException.catching(() -> {
      // First check the we have at least the minimum number of arguments
      if (argumentTypes.size() < arguments.size() - 1 || argumentTypes.size() > arguments.size()) {
        throw new ProblemException(ArgsProblems.get()
            .wrongNumberRange(arguments.size() - 1, arguments.size(), argumentTypes.size()));
      }

      Boolean nullable = Nullable.is(argumentTypes.get(0));
      ContinuousFunctionType curveType =
          Types.findOrThrow("function", ContinuousFunctionType.ANY, Nullable.strip(argumentTypes.get(0)));

      // now check the right number of arguments expected for the curve we have
      if (argumentTypes.size() != curveType.getDimensions() + 1) {
        throw new ProblemException(ArgsProblems.get().wrongNumber(curveType.getDimensions() + 1, argumentTypes.size()));
      }

      // we now check that all the dimension arguments are numbers
      for (int i = 0; i < curveType.getDimensions(); i++) {
        Type givenValueType = argumentTypes.get(i + 1);
        if (Nullable.is(givenValueType) || !(Number.class.isAssignableFrom(givenValueType.internalType()))) {
          throw new ProblemException(
              ArgsProblems.mismatch(arguments.get(i + 1), givenValueType)
          );
        }
      }

      return new FunctionImpl(curveType, argumentTypes, Nullable.ifTrue(nullable, curveType.getReturnType()));
    });
  }

  public RiskscapeFunction asFunction() {
    return RealizableFunction.asFunction(this, arguments, Types.FLOATING);
  }
}
