/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.rl.agg.Accumulator;
import nz.org.riskscape.engine.rl.agg.AggregationFunction;
import nz.org.riskscape.engine.rl.agg.RealizedAggregateExpression;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.RelationType;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.util.RelationBackedList;
import nz.org.riskscape.engine.util.SegmentedList;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

/**
 * Aggregation function that simply returns all the collected elements in a list.  The implementation uses a
 * SegmentedList to efficiently store an unknown number of elements in a list.
 */
public class ToListFunction extends BaseRealizableFunction implements AggregationFunction {

  public ToListFunction() {
    super(ArgumentList.create("value", Types.ANYTHING), RSList.create(Types.ANYTHING));
  }

  @SuppressWarnings({"rawtypes", "unchecked"})
  private static class AccumImpl implements Accumulator {

    private final RealizedExpression expression;
    private final List list;

    AccumImpl(RealizedExpression expression) {
      this.expression = expression;
      list = SegmentedList.forType(expression.getResultType());
    }

    @Override
    public Accumulator combine(Accumulator rhs) {
      List rhsList = ((AccumImpl)rhs).list;
      list.addAll(rhsList);
      return this;
    }

    @Override
    public void accumulate(Object input) {
      list.add(expression.evaluate(input));
    }

    @Override
    public Object process() {
      return list;
    }

    @Override
    public boolean isEmpty() {
      return false;
    }
  }

  @Override
  public RiskscapeFunction asFunction() {
    return AggregationFunction.addAggregationTo(this, super.asFunction());
  }

  // aggregate version
  @Override
  public ResultOrProblems<RealizedAggregateExpression> realize(RealizationContext context, Type inputType,
      FunctionCall fc) {

    return arguments.getRequiredArgument(fc, "value").flatMap(arg ->
      context.getExpressionRealizer().realize(inputType, arg.getExpression())
    ).map(realized -> {
      Type resultType = RSList.create(realized.getResultType());
      return RealizedAggregateExpression.create(inputType, resultType, fc, () -> new AccumImpl(realized));
    });
  }

  // scalar version (turns a relation into a list)
  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {
    return ProblemException.catching(() -> {
      if (givenTypes.size() != 1) {
        throw new ProblemException(ArgsProblems.get().wrongNumber(1, givenTypes.size()));
      }
      RelationType relationType = givenTypes.get(0).findAllowNull(RelationType.class)
          .orElseThrow(() -> new ProblemException(
              TypeProblems.get().mismatch(arguments.get(0), RelationType.WILD, givenTypes.get(0))
          ));

      return RiskscapeFunction.create(this, givenTypes, RSList.create(relationType.getContainedType()),
          args -> new RelationBackedList((Relation) args.get(0), context.getProblemSink()));
    });
  }
}
