/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import java.util.ArrayList;
import java.util.List;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

import lombok.RequiredArgsConstructor;

public class SwitchFunction extends BaseRealizableFunction {

  /*
   * Implementation notes:
   *
   * This is fairly rough rugged and row with the way it validates and massages the types etc, we could make
   * more use of the new ancestor rules to be more permissive with what it accepts by coercing values for you
   */

  public interface LocalProblems extends ProblemFactory {
    Problem caseOptionsMustBeSameTypeAsOn(Type onType, Type caseOptionType);
    Problem caseMustReturnSameTypeAsDefault(Type defaultType, Type caseType);
  }

  public static final Struct CASE_TYPE = Struct.of("in", RSList.LIST_ANYTHING, "return", Types.ANYTHING);
  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  @RequiredArgsConstructor
  private static class Case {
    final List<Object> values;
    final Object result;

    boolean matches(Object value) {
      return values.contains(value);
    }
  }

  public SwitchFunction() {
    super(
        ArgumentList.create(
            "on", Types.ANYTHING,
            "default", Types.ANYTHING,
            "cases", RSList.create(CASE_TYPE)
        ),
        Types.ANYTHING
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {
    // All arguments are required
    if (givenTypes.size() != arguments.size()) {
      return ResultOrProblems.failed(ArgsProblems.get().wrongNumber(arguments.size(), givenTypes.size()));
    }

    boolean nullableReturnType = Nullable.is(givenTypes.get(1));
    Type baseReturnType = Nullable.strip(givenTypes.get(1));

    Struct caseType = givenTypes.get(2).find(RSList.class)
        .map(RSList::getContainedType)
        .flatMap(c -> c.find(Struct.class))
        .orElse(null);
    if (caseType == null
        || !(caseType.size() == 2 && caseType.hasMember("in") && caseType.hasMember("return"))) {
      return ResultOrProblems.failed(ArgsProblems.mismatch(arguments.get(2), givenTypes.get(2)));
    }

    Type listType = caseType.getEntry("in").getType()
        .find(RSList.class)
        .map(RSList::getContainedType)
        .orElse(null);

    // we strip any nullables from onType to make the checks with the case/in values work
    Type onType = Nullable.strip(givenTypes.get(0));
    // the return type will be nullable if the on type is nullable
    nullableReturnType = nullableReturnType || Nullable.is(givenTypes.get(0));

    if (onType != listType) {
      // the list type should be the same as the on type
      return ResultOrProblems.failed(PROBLEMS.caseOptionsMustBeSameTypeAsOn(onType, listType));
    }

    Type caseReturnType = caseType.getEntry("return").getType();
    nullableReturnType = nullableReturnType || Nullable.is(caseReturnType);
    if (baseReturnType != Nullable.strip(caseReturnType)) {
      return ResultOrProblems.failed(PROBLEMS.caseMustReturnSameTypeAsDefault(
          baseReturnType, caseType.getEntry("return").getType()));
    }

    Type actualReturnType = Nullable.ifTrue(nullableReturnType, baseReturnType);

    return getArguments().evaluateConstant(context, functionCall, "cases", List.class, CASE_TYPE).flatMap(values -> {
      List<Case> cases = new ArrayList<>();
      for (Object value: values) {
        Tuple tuple = (Tuple)value;
        @SuppressWarnings("unchecked")
        List<Object> in = (List<Object>)tuple.fetch("in");
        cases.add(new Case(in, tuple.fetch("return")));
      }

      return ResultOrProblems.of(RiskscapeFunction.create(this, givenTypes, actualReturnType, args -> {
        Object on = args.get(0);
        if (on == null) {
          return null;
        }
        for (Case candidate : cases) {
          if (candidate.matches(on)) {
            return candidate.result;
          }
        }

        // return the default
        return args.get(1);
      }));
    });
  }

}
