/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.Point;

import com.google.common.collect.Lists;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.coverage.TypedCoverage;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.geo.GeometryUtils;
import nz.org.riskscape.engine.geo.ProjectGeometryOp;
import nz.org.riskscape.engine.geo.RecursiveQuadGridOp;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.CoverageType;
import nz.org.riskscape.engine.types.GeomType;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class SegmentByGrid extends BaseRealizableFunction {

  public SegmentByGrid() {
    super(
      ArgumentList.create("geometry", Types.GEOMETRY, "distance", Types.FLOATING, "align-to", Types.ANYTHING),
      RSList.create(Types.GEOMETRY)
    );
  }

  public interface LocalProblems extends ProblemFactory {
    Problem coverageHasNoEnvelope(TypedCoverage coverage);
  }

  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> types) {

    if (functionCall.getArguments().size() != arguments.size()) {
      return ResultOrProblems.failed(ArgsProblems.get()
          .wrongNumber(arguments.size(), functionCall.getArguments().size()));
    }

    SRIDSet sridSet = context.getProject().getSridSet();
    RecursiveQuadGridOp cutByGridOp = new RecursiveQuadGridOp();

    Optional<GeomType.Point> pointType = types.get(2).findAllowNull(GeomType.Point.class);
    Optional<CoverageType> coverageType = types.get(2).findAllowNull(CoverageType.class);

    if (! pointType.isPresent() && ! coverageType.isPresent()) {
      return ResultOrProblems.failed(ArgsProblems.mismatch(arguments.get(2), types.get(2),
          Lists.newArrayList(Types.POINT, CoverageType.WILD)));
    }

    return ResultOrProblems.of(RiskscapeFunction.create(this, arguments.getArgumentTypes(), returnType, args -> {
      Geometry toSegment = (Geometry) args.get(0);
      Double distance = (Double) args.get(1);
      Point alignTo;
      Object alignToObject = args.get(2);
      if (alignToObject instanceof TypedCoverage) {
        TypedCoverage coverage = (TypedCoverage)alignToObject;
        alignTo = coverage.getEnvelope()
            .map(env -> sridSet.getGeometryFactory(
                coverage.getCoordinateReferenceSystem()).createPoint(new Coordinate(env.getMinX(), env.getMinY())))
            .orElseThrow(() -> new RiskscapeException(PROBLEMS.coverageHasNoEnvelope(coverage)));
      } else {
        alignTo = (Point)alignToObject;
      }

      ProjectGeometryOp projectOp = new ProjectGeometryOp(sridSet);
      return projectOp.apply(toSegment).stream()
          .flatMap(projected -> {
            List<Geometry> segmented = new ArrayList<>();
            // make sure the grid origin is in the same projected CRS we're working in
            Point origin = (Point) projected.projectAlso(alignTo);
            GeometryUtils.processPerPart(projected.getProjected(),
                geomPart -> segmented.addAll(cutByGridOp.apply(geomPart, distance, origin))
            );
            // Ensure that the segmented geometries are reprojectred back to the source CRS
            return segmented.stream()
                .map(g -> projected.toSourceCrs(g));
          })
          .collect(Collectors.toList());

    }));
  }

}
