/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.function.BiFunction;
import java.util.stream.Collectors;

import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryCollection;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.Point;

import nz.org.riskscape.defaults.geo.SegmentLineStringOp;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.geo.ProjectGeometryOp;
import nz.org.riskscape.engine.geo.RecursiveQuadGridOp;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class Segment extends BaseRealizableFunction {

  public Segment() {
    super(ArgumentList.create("geometry", Types.GEOMETRY, "distance", Types.FLOATING), RSList.create(Types.GEOMETRY));
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> types) {

    SRIDSet sridSet = context.getProject().getSridSet();
    SegmentLineStringOp op = new SegmentLineStringOp();
    RecursiveQuadGridOp quadGridOp = new RecursiveQuadGridOp();

    BiFunction<Geometry, Double, List<? extends Geometry>> nonCollection = (geom, distance) -> {
      if (geom instanceof LineString) {
        return op.apply((LineString) geom, distance);
      } else if (geom instanceof Point) {
        return Collections.singletonList(geom);
      } else {
        return quadGridOp.apply(geom, distance);
      }
    };

    BiFunction<GeometryCollection, Double, List<? extends Geometry>> collection = (geom, distance) -> {
      int size = geom.getNumGeometries();
      List<List<? extends Geometry>> chunks = new ArrayList<>(size);
      int totalSize = 0;
      for (int i = 0; i < size; i++) {
        Geometry child = geom.getGeometryN(i);
        @SuppressWarnings("unchecked")
        List<Geometry> childSegments = (List<Geometry>) nonCollection.apply(child, distance);
        chunks.add(childSegments);
        totalSize += childSegments.size();
      }

      List<Geometry> collected = new ArrayList<>(totalSize);
      for (List<? extends Geometry> list : chunks) {
        collected.addAll(list);
      }

      return collected;
    };

    Type myReturnType = returnType;
    if (types.size() > 0) {
      // we attempt to get the referenced crs from the first argument type so we can include this in
      // the return type. Note that we don't try to use a more specific geometry type because the segmenting
      // could turn polygons to multi polygons etc.
      myReturnType = types.get(0).findAllowNull(Referenced.class)
          .map(Referenced::getCrs)
          .map(crs -> (Type)RSList.create(Referenced.of(Types.GEOMETRY, crs)))
          .orElse(myReturnType);
    }

    return ResultOrProblems.of(RiskscapeFunction.create(this, arguments.getArgumentTypes(), myReturnType, args -> {
      Geometry toSegment = (Geometry) args.get(0);
      Double distance = (Double) args.get(1);

      ProjectGeometryOp projectOp = new ProjectGeometryOp(sridSet);
      return projectOp.apply(toSegment).stream()
          .flatMap(projected -> {
            List<? extends Geometry> segmented = null;
            if (projected.getProjected() instanceof GeometryCollection) {
              segmented = collection.apply((GeometryCollection) projected.getProjected(), distance);
            } else {
              segmented = nonCollection.apply(projected.getProjected(), distance);
            }
            // Ensure that the segmented geometries are reprojectred back to the source CRS
            return segmented.stream()
                .map(g -> projected.toSourceCrs(g));
          })
          .collect(Collectors.toList());
    }));
  }

}
