/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import java.util.List;

import lombok.Getter;
import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.lookup.LookupTable;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.LookupTableType;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

/**
 * Returns a value from a lookup table
 */
public class LookupFunction implements RealizableFunction, RiskscapeFunction {

  @Getter
  private final Type returnType = Types.ANYTHING;

  @Getter
  private final ArgumentList arguments = ArgumentList.create(
      "lookuptable", LookupTableType.WILD,
      "key", Types.ANYTHING
  );

  @Getter
  private final List<Type> argumentTypes = arguments.getArgumentTypes();

  @Override
  public Object call(List<Object> args) {
    throw new UnsupportedOperationException();
  }
  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {

    return ProblemException.catching(() -> {
      if (givenTypes.size() != 2) {
        throw new ProblemException(ArgsProblems.get().wrongNumber(2, givenTypes.size()));
      }

      LookupTableType tableType = Types.findOrThrow(functionCall, LookupTableType.WILD, givenTypes.get(0));

      Type givenKeyType = givenTypes.get(1);
      if (!context.getProject().getTypeSet().isAssignable(givenKeyType, tableType.getKeyType())) {
        throw new ProblemException(TypeProblems.get().mismatch(functionCall, tableType.getKeyType(), givenKeyType));
      }

      return new RiskscapeFunction() {

        @Override
        public Type getReturnType() {
          return Nullable.of(tableType.getValueType());
        }

        @Override
        public List<Type> getArgumentTypes() {
          return givenTypes;
        }

        @Override
        public Object call(List<Object> args) {
          LookupTable table = (LookupTable) args.get(0);
          return table.lookup(args.get(1));
        }
      };
    });
  }


}
