/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.function;

import java.util.Arrays;
import java.util.List;

import com.google.common.collect.Range;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.BaseRealizableFunction;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructBuilder;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.ProblemException;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

@Deprecated
public class ListToTuple extends BaseRealizableFunction {

  public ListToTuple() {
    super(
        ArgumentList.create(
            "list", Nullable.of(RSList.LIST_ANYTHING),
            "upto", Types.INTEGER,
            "prefix", Nullable.TEXT
        ),
        Types.ANYTHING
    );
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> givenTypes) {
    return super.realize(context, functionCall, givenTypes)
        .withMoreProblems(GeneralProblems.get().deprecated("list_to_tuple", "list_to_columns"));
  }

  @Override
  public RiskscapeFunction build(RealizationContext context, FunctionCall functionCall,
      List<Type> argumentTypes) throws ProblemException {
    RSList listType = arguments.getRequiredAs(argumentTypes, 0, RSList.class).getOrThrow();

    Long upTo = arguments.get("upto").evaluateConstant(context, functionCall, Long.class).getOrThrow();
    String prefix = "";
    if (argumentTypes.size() == 3) {
      prefix = arguments.get("prefix").evaluateConstant(context, functionCall, String.class).getOrThrow();
    }

    if (upTo < 1) {
      throw new ProblemException(GeneralProblems.get()
          .valueOutOfRange(arguments.getArgument(functionCall, "upto").get(), upTo, Range.atLeast(1L)));
    }

    // struct member type is nullable because we don't know if the list will have as many items
    // as upTo requires
    Type structMemberType = Nullable.of(listType.getContainedType());
    StructBuilder builder = new StructBuilder();
    for (int i = 0; i < upTo; i++) {
      builder.add(prefix + i, structMemberType);
    }
    Struct result = builder.build();

    return RiskscapeFunction.create(this, argumentTypes, result, args -> {
      List<?> values = (List<?>) args.get(0);
      if (values == null) {
        return new Tuple(result);
      } else {
        Object[] valueArray = values.toArray();
        if (valueArray.length > upTo) {
          // if the value list has more items than required we need to truncate it otherwise the
          // Tuple will blow up
          valueArray = Arrays.copyOf(valueArray, upTo.intValue());
        }
        return Tuple.ofValues(result, valueArray);
      }
    });
  }

}
