/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.curves;

import java.util.List;
import java.util.Map;

import org.apache.commons.math3.analysis.ParametricUnivariateFunction;
import org.apache.commons.math3.analysis.polynomials.PolynomialFunction;
import org.apache.commons.math3.exception.ConvergenceException;
import org.apache.commons.math3.exception.MathIllegalArgumentException;
import org.apache.commons.math3.fitting.SimpleCurveFitter;
import org.apache.commons.math3.util.FastMath;

import lombok.Getter;
import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.BoundParameters;
import nz.org.riskscape.engine.bind.JavaParameterSet;
import nz.org.riskscape.engine.function.UntypedFunction;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Fits a power law curve to the data using the method outlined in
 * https://scipy-cookbook.readthedocs.io/items/FittingData.html#Fitting-a-power-law-to-data-with-errors
 */
public class PowerLawFitter implements CurveFitter<double[]> {

  private static final ParametricUnivariateFunction POLYNOMIAL = new PolynomialFunction.Parametric();

  public static class Params {
    public double linearYIntercept = 1D;
    public double linearSlope = -1D;
  }

  @Override
  public String getId() {
    return "powerlaw";
  }

  @Override
  public Class<double[]> getBindingClass() {
    return double[].class;
  }

  @Override
  public ResultOrProblems<double[]> bind(BindingContext context, Map<String, List<?>> unbound) {
    BoundParameters bound = parameters.bind(context, unbound);
    Params params = parameters.bindToObject(bound).getBoundToObject();

    return ResultOrProblems.of(new double[] {
        params.linearYIntercept,
        params.linearSlope
    });
  }

  @Getter
  public JavaParameterSet<Params> parameters = JavaParameterSet.fromBindingClass(Params.class);

  @Override
  public FitCurve<double[]> fit(double[] startParams, ObservedPoints points) {
    SimpleCurveFitter fitter = SimpleCurveFitter.create(POLYNOMIAL, startParams);
    double[] fitParams;
    try {
      fitParams = fitter.fit(points.withLogXLogY());
    } catch (ConvergenceException | MathIllegalArgumentException ex) {
      throw new CannotFitCurveException(Problems.caught(ex));
    }
    return new DefaultFitCurve<>(this, getFunction(fitParams), fitParams, points, CurveFitter::computeRSquared);
  }

  public UntypedFunction getFunction(double[] coefficients) {
    return args -> {
      double x = ((Number) args.get(0)).doubleValue();
      return Math.exp(POLYNOMIAL.value(FastMath.log(x), coefficients));
    };
  }

}
