/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.curves;

import java.util.AbstractList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Stream;

import org.apache.commons.math3.fitting.WeightedObservedPoint;
import org.apache.commons.math3.util.FastMath;

import com.google.common.collect.Lists;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.util.Pair;

@RequiredArgsConstructor
public class ObservedPoints extends AbstractList<WeightedObservedPoint> {

  private final List<Double> xValues;
  private final List<Double> yValues;

  @Override
  public Iterator<WeightedObservedPoint> iterator() {
    return new IterImpl(xValues.iterator(), yValues.iterator());
  }

  @RequiredArgsConstructor
  private static class IterImpl implements Iterator<WeightedObservedPoint> {
    private final Iterator<Double> xValueIter;
    private final Iterator<Double> yValueIter;

    @Override
    public WeightedObservedPoint next() {
      Double x = xValueIter.next();
      Double y = yValueIter.next();

      return new WeightedObservedPoint(1D, x, y);
    }

    @Override
    public boolean hasNext() {
      return xValueIter.hasNext();
    }
  }

  @Override
  public int size() {
    return xValues.size();
  }

  @Override
  public WeightedObservedPoint get(int index) {
    Double x = xValues.get(index);
    Double y = yValues.get(index);

    return new WeightedObservedPoint(1D, x, y);
  }

  public List<WeightedObservedPoint> withLogX() {
    return Lists.transform(this, point ->
      new WeightedObservedPoint(point.getWeight(), FastMath.log(point.getX()), point.getY())
    );
  }

  public List<WeightedObservedPoint> withLogY() {
    return Lists.transform(this, point ->
      new WeightedObservedPoint(point.getWeight(), point.getX(), FastMath.log(point.getY()))
    );
  }

  public List<WeightedObservedPoint> withLogXLogY() {
    return Lists.transform(this, point ->
      new WeightedObservedPoint(point.getWeight(), FastMath.log(point.getX()), FastMath.log(point.getY()))
    );
  }

  /**
   * @return a view of this list as pairs.  WeightedObservedPoint is missing many niceities, like equals, or toString,
   * which can be useful/important.
   */
  public List<Pair<Double, Double>> asListOfPairs() {
    return Lists.transform(this, point -> Pair.of(point.getX(), point.getY()));
  }

  /**
   * @return a {@link Stream} of all the y values in the set of observed points.  This can be a 'cheaper' operation than
   * getting the list of WeightedObservedPoints and streaming through those.
   */
  public Stream<Double> getYValues() {
    return yValues.stream();
  }

  /**
   * @return a {@link Stream} of all the x values in the set of observed points.  This can be a 'cheaper' operation than
   * getting the list of WeightedObservedPoints and streaming through those.
   */
  public Stream<Double> getXValues() {
    return xValues.stream();
  }

}
