/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.curves;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import nz.org.riskscape.engine.bind.BindingContext;
import nz.org.riskscape.engine.bind.Parameter;
import nz.org.riskscape.engine.bind.ParameterSet;
import nz.org.riskscape.engine.function.UntypedFunction;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.util.Pair;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

public class ContinuousLinearFitter implements CurveFitter<Double> {

  public interface LocalProblems extends ProblemFactory {
    Problem noData();

    Problem duplicateXValues(String id, Double value);
  }

  public static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  @Override
  public String getId() {
    return "continuous";
  }

  @Override
  public ParameterSet getParameters() {
    return ParameterSet.from(Parameter.required("score", Double.class, 0D));
  }

  @Override
  public ResultOrProblems<Double> bind(BindingContext context, Map<String, List<?>> unbound) {
    return getParameters().bind(context, unbound)
        .flatMap(bound -> ResultOrProblems.of((Double)bound.getValue("score")));
  }

  @Override
  public Class<Double> getBindingClass() {
    return Double.class;
  }

  @Override
  public FitCurve<Double> fit(Double score, ObservedPoints points) {

    List<Pair<Double, Double>> pairs = new ArrayList<>(points.asListOfPairs());
    pairs.sort((a, b) -> a.getLeft().compareTo(b.getLeft()));

    if (pairs.isEmpty()) {
      throw new CannotFitCurveException(PROBLEMS.noData());
    }

    // sanity-check there are no duplicate X values
    Set<Double> unique = new HashSet<>();
    for (Pair<Double, Double> pair : pairs) {
      if (!unique.add(pair.getLeft())) {
        throw new CannotFitCurveException(PROBLEMS.duplicateXValues(getId(), pair.getLeft()));
      }
    }

    double[] xsAndYs = new double[pairs.size() * 2];
    int i = 0;
    for (Pair<Double,Double> pair : pairs) {
      xsAndYs[i++] = pair.getLeft();
      xsAndYs[i++] = pair.getRight();
    }

    // TODO allow a set score as it will always "fit" perfectly
    return new DefaultFitCurve<>(this, getFunction(xsAndYs), score, points, x -> score);
  }

  public UntypedFunction getFunction(double[] xsAndYs) {
    return args -> {
      double x = ((Number) args.get(0)).doubleValue();
      for (int i = 0; i < xsAndYs.length - 2;) {
        double x1 = xsAndYs[i++];   // we consume x1 and y2
        double y1 = xsAndYs[i++];
        double x2 = xsAndYs[i];     // and look ahead to get x2 and y2
        double y2 = xsAndYs[i + 1];

        if (x >= x1 && x < x2) {
          double slope = (y2 - y1) / (x2 - x1);
          double yIntercept = y1 - slope * x1;

          return slope * x + yIntercept;
        }
      }

      // out of bounds of weighted points, return limits
      if (x < xsAndYs[0]) {
        // return first y
        return xsAndYs[1];
      } else {
        // return last y
        return xsAndYs[xsAndYs.length - 1];
      }
    };
  }

}
