/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import static nz.org.riskscape.defaults.classifier.ProblemCodes.*;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;

import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.eqrule.Coercer;
import nz.org.riskscape.engine.typeset.TypeSet;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ResultOrProblems;
import lombok.Getter;
import lombok.RequiredArgsConstructor;

/**
 * Wraps a {@link RealizedTreeFilter} so that the optimal {@link #getResultType()} can be produced
 * from the many {@link RealizedTreeExpression}s that could be called upon when
 * {@link #evaluate(java.lang.Object)} is called.
 *
 * If we actually cared about classifier functions, we could drop this class and just use typeSet.computeAncestors
 */
@RequiredArgsConstructor
public class ReturnTypeInferer {

  public static ResultOrProblems<ReturnTypeInferer> build(RealizedTreeFilter filter, TypeSet typeSet) {
    List<RealizedTreeExpression> collectedExpressions = filter.collectRealizedExpressions();

    return inferType(filter, collectedExpressions, typeSet)
        .map(returnType -> {
          Map<RealizedTreeExpression, Coercer> coercers = new HashMap<>();
          for (RealizedTreeExpression treeExpression : collectedExpressions) {
            if (!typeSet.isAssignable(treeExpression.getResultType(), returnType)) {
              // Coercion is required
              Coercer coercer = typeSet.findEquivalenceCoercer(treeExpression.getResultType(), returnType)
                  .orElseThrow(() -> new NoSuchElementException());
              coercers.put(treeExpression, coercer);
            }
          }
          return new ReturnTypeInferer(filter, returnType, coercers);
        });
  }

  private static ResultOrProblems<Type> inferType(
      RealizedTreeFilter filter,
      List<RealizedTreeExpression> collectedExpressions,
      TypeSet typeSet) {

    if (filter.getChildren().size() == 0) {
      return ResultOrProblems.of(filter.getOrElse().resultType);
    }

    // make sure everything is a struct or not a struct
    RealizedTreeExpression firstExpression = collectedExpressions.get(0);
    boolean wasStruct = firstExpression.getResultType().find(Struct.class).isPresent();
    for (RealizedTreeExpression re : collectedExpressions) {
      if (re == RealizedTreeExpression.EMPTY) {
        continue;
      }

      boolean alsoStruct = re.getResultType().find(Struct.class).isPresent();
      if (alsoStruct != wasStruct) {

        return ResultOrProblems.failed(Problem.error(MIXED_TREE_RESULT_TYPES,
            firstExpression.expression.getIdentifier().getLocation().getLine(),
            re.expression.getIdentifier().getLocation().getLine()));
      }
    }

    boolean hasNoDefault = filter.getOrElse().equals(RealizedTreeExpression.EMPTY);
    Type pointer = null;
    for (RealizedTreeExpression rte : collectedExpressions) {
      if (pointer == null) {
        pointer = rte.getResultType();
      } else {
        pointer = findBestReceiverType(pointer, rte.getResultType(), typeSet);
      }
    }
    return ResultOrProblems.of(Nullable.ifTrue(hasNoDefault, pointer));
  }

  /**
   * Finds the best receiver type between the two input {@link Type}s. Note that best match does not  necessarily
   * mean that either input type can be assigned to the best match. Coercion may be required.
   *
   * This method checks the following in order:
   * - assignment b -> a, a is best match
   * - assignment a -> b, b is best match
   * - coercible b -> a, a is best
   * - coercible a -> b, b is best
   * - else anything
   *
   * @param a first type to check
   * @param b second type to check
   * @param typeSet
   * @return the best match between the two types.
   */
  static Type findBestReceiverType(Type a, Type b, TypeSet typeSet) {
    // the resulting type will be nullable if either input type is nullable
    boolean nullable = Nullable.is(a) || Nullable.is(b);

    Type previousBestMatch = Nullable.strip(a);
    Type currentBestMatch = previousBestMatch;
    Type toMatch = Nullable.strip(b);

    Optional<Struct> currentBestStruct = currentBestMatch.find(Struct.class);
    Optional<Struct> toMatchStruct = toMatch.find(Struct.class);

    if (currentBestStruct.isPresent() && toMatchStruct.isPresent()) {
      // structs get merged
      currentBestMatch = merge(currentBestStruct.get(), toMatchStruct.get(), typeSet);
    } else if (!typeSet.isAssignable(toMatch, currentBestMatch)) {
      // cannot assign toMatch to currentBestMatch. Maybe assignment will work the other way around
      if (typeSet.isAssignable(currentBestMatch, toMatch)) {
        currentBestMatch = toMatch;
      } else {
        if (!typeSet.findEquivalenceCoercer(toMatch, currentBestMatch).isPresent()) {
          // to match cannot be coerced to current best match
          if (typeSet.findEquivalenceCoercer(currentBestMatch, toMatch).isPresent()) {
            // but currect best match can be coerced to toMatch, toMatch is best
            currentBestMatch = toMatch;
          } else {
            if (currentBestMatch.isWrapped() || toMatch.isWrapped()) {
              // let's strip the wrapping types then look for best match
              currentBestMatch = findBestReceiverType(currentBestMatch.getUnwrappedType(), toMatch.getUnwrappedType(),
                  typeSet);
            } else {
              // there is no match, anything goes
              currentBestMatch = Types.ANYTHING;
            }
          }
        }
      }
    }
    return Nullable.ifTrue(nullable, currentBestMatch);
  }

  /**
   * Merges two {@link Struct}s together to get the combined attributes. Any attributes that
   * exist in one (but not the other) will be made {@link Nullable}. Attribute types may also be optimized
   * by applying {@link #findBestReceiverType(Type, Type, TypeSet) }.
   */
  static Struct merge(Struct startWith, Struct toMerge, TypeSet typeSet) {
    Struct merged = startWith;
    for (StructMember thisMember : toMerge.getMembers()) {
      StructMember collatedMember = merged.getEntry(thisMember.getKey());
      if (collatedMember == null) {
        merged = merged.add(thisMember.getKey(), Nullable.of(thisMember.getType()));
      } else {
        Type bestMatch = findBestReceiverType(
            collatedMember.getType(),
            thisMember.getType(),
            typeSet
        );

        merged = merged.addOrReplace(thisMember.getKey(), bestMatch);
      }
    }

    // now we need to check for members that are in merged but not in a and make them nullalble
    for (StructMember thisMember: merged.getMembers()) {
      if (!toMerge.getMember(thisMember.getKey()).isPresent()) {
        merged = merged.addOrReplace(thisMember.getKey(), Nullable.of(thisMember.getType()));
      }
    }
    return merged;
  }

  @Getter
  private final RealizedTreeFilter wrapped;
  @Getter
  private final Type resultType;
  private final Map<RealizedTreeExpression, Coercer> coercers;

  public Object evaluate(Object input) {
    RealizedTreeExpression expression = wrapped.match(input);

    if (expression == null) {
      return null;
    }
    Object result = expression.evaluate(input);
    Coercer coercer = coercers.get(expression);
    if (coercer != null) {
      result = coercer.apply(result);
    }
    return result;
  }

  boolean isDefaultPresent() {
    return wrapped.isDefaultPresent();
  }

}
