/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import static nz.org.riskscape.defaults.classifier.ProblemCodes.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

import lombok.Getter;
import lombok.RequiredArgsConstructor;

import nz.org.riskscape.defaults.classifier.AST.FunctionDecl;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

@RequiredArgsConstructor
public class RealizedTreeFilter {

  public static ResultOrProblems<RealizedTreeFilter> build(ExpressionRealizer realizer, FunctionDecl function,
      Type inputType) {

    List<Problem> problems = new ArrayList<>();

    if (function.body.isEmpty() && !function.defaultExpr.isPresent()) {
      problems.add(Problem.error(EMPTY_BODY));
      return ResultOrProblems.failed(problems);
    }

    RealizedTreeExpression defaultExpr = RealizedTreeExpression.EMPTY;
    if (function.defaultExpr.isPresent()) {
      ResultOrProblems<RealizedTreeExpression> exprOr =
          RealizedTreeExpression.build(realizer, function.defaultExpr.get(), inputType);

      if (exprOr.hasProblems()) {
        problems.add(Problems.foundWith(function.defaultExpr.get().getIdentifier(), exprOr.getProblems()));
      }
      defaultExpr = exprOr.orElse(RealizedTreeExpression.EMPTY);
    }

    List<RealizedTreeFilter> realizedFilters = new ArrayList<RealizedTreeFilter>();
    for (AST.Filter filterAst: function.body) {
      ResultOrProblems<RealizedTreeFilter> filterOr = build(realizer, filterAst, inputType);
      if (filterOr.hasProblems()) {
        problems.add(Problems.foundWith(filterAst.getIdentifier(), filterOr.getProblems()));
      }
      if (filterOr.isPresent()) {
        realizedFilters.add(filterOr.getWithProblemsIgnored());
      }
    }

    RealizedTreeFilter rtf = new RealizedTreeFilter(
        null,
        RealizedExpression.TRUE,
        defaultExpr,
        realizedFilters
    );

    return ResultOrProblems.of(rtf, problems);
  }

  public static ResultOrProblems<RealizedTreeFilter> build(
      ExpressionRealizer realizer,
      AST.Filter filter,
      Type inputType
  ) {
    List<RealizedTreeFilter> children = new ArrayList<>(filter.children.size());
    List<Problem> filterProblems = new ArrayList<>();

    if (filter.children.isEmpty() && ! filter.orElse.isPresent()) {
      // a filter cannot be empty
      filterProblems.add(Problem.error(EMPTY_FILTER, filter.identifier.getLocation().getLine()));
    }

    for (AST.Filter childFilter : filter.children) {
      ResultOrProblems<RealizedTreeFilter> filterOr = build(realizer, childFilter, inputType);

      if (filterOr.hasProblems()) {
        filterProblems.add(Problems.foundWith(childFilter.getIdentifier(), filterOr.getProblems()));
      }
      if (filterOr.isPresent()) {
        children.add(filterOr.get());
      }
    }

    ResultOrProblems<RealizedExpression> expressionOr = realizer.realize(inputType, filter.built);
    ResultOrProblems<RealizedTreeExpression> funcOr = filter.orElse
        .map(e -> RealizedTreeExpression.build(realizer, e, inputType))
        .map(exprOr -> {
          if (exprOr.hasProblems()) {
            filterProblems.add(Problems.foundWith(filter.orElse.get().getIdentifier(), exprOr.getProblems()));
          }
          return exprOr;
        })
        .orElse(null);

    if (expressionOr.isPresent()) {
      return ResultOrProblems.of(new RealizedTreeFilter(
          filter,
          expressionOr.get(),
          funcOr == null ? null : funcOr.orElse(null),
          children
      ), filterProblems);
    } else {
      filterProblems.add(Problems.foundWith(filter.built, expressionOr.getProblems()));
      return ResultOrProblems.failed(filterProblems);
    }
  }

  static final RealizedTreeFilter fallback(RealizedTreeExpression yield) {
    return new RealizedTreeFilter(null, RealizedExpression.TRUE, yield, Collections.emptyList());
  }

  private final AST.Filter filter;
  @Getter
  private final RealizedExpression condition;

  // applied when condition matches, but none of the children do
  @Getter
  private final RealizedTreeExpression orElse;
  @Getter
  private final List<RealizedTreeFilter> children;

  // TODO calculate from visit
  private int maxDepth = 10;
  private boolean exclusive = true;

  public Object evaluate(Object input) {
    RealizedTreeExpression expression = match(input);

    if (expression == null) {
      return null;
    } else {
      return expression.evaluate(input);
    }
  }

  // N.b. this method is implemented without recursion to improve performance - match is likely to be called a lot
  // during execution
  public RealizedTreeExpression match(Object criteria) {
    ArrayList<RealizedTreeFilter> visitStack = new ArrayList<>(maxDepth);

    visitStack.add(this);

    while (!visitStack.isEmpty()) {
      RealizedTreeFilter ptr = visitStack.remove(0);
      if (ptr.condition.evaluate(criteria) == Boolean.TRUE) {

        if (ptr.children.isEmpty()) {
          return ptr.orElse;
        } else {
          // TODO if filtering is exclusive, we should drop all other elements from the queue here to avoid those being
          // visited
          if (exclusive) {
            visitStack.clear();
          }

          visitStack.addAll(ptr.children);
          if (ptr.orElse != null) {
            // if none of the children match, this will and return the orElse expression
            // TODO add this to the list of children at the end to avoid allocation every time match is invoked
            visitStack.add(RealizedTreeFilter.fallback(ptr.orElse));
          }
        }
      }
    }

    return orElse;
  }

  /**
   * @return true if this tree will always give a match, e.g. there's a default case covering every branch of the tree.
   * If this returns false, then any return type built from this tree should be nullable.
   */
  boolean isDefaultPresent() {
    return orElse != null && orElse != RealizedTreeExpression.EMPTY;
  }

  @Override
  public String toString() {
    if (children.size() == 0) {
      return String.format("RealizedTreeFilter(if=%s, then=%s)", this.condition, this.orElse);
    } else {
      return String.format(
          "RealizedTreeFilter(if=%s, nested=%s, default=%s)",
          this.condition,
          this.children,
          this.orElse
      );
    }
  }

  public List<RealizedTreeExpression> collectRealizedExpressions() {
    LinkedList<RealizedTreeFilter> stack = new LinkedList<>();
    stack.add(this);
    List<RealizedTreeExpression> collected = new ArrayList<>();
    while (!stack.isEmpty()) {
      RealizedTreeFilter ptr = stack.removeFirst();

      stack.addAll(ptr.children);

      if (ptr.orElse != null && ptr.orElse != RealizedTreeExpression.EMPTY) {
        collected.add(ptr.orElse);
      }
    }

    return collected;
  }

}
