/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import java.util.ArrayList;
import java.util.List;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.RealizedExpression;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructBuilder;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * Wraps an expression from a classifier to deal with either simple or struct type expressions.  Used by pre, post
 * default and function parts of classifiers.
 */
@RequiredArgsConstructor
public abstract class RealizedTreeExpression {

  public static final RealizedTreeExpression EMPTY = new RealizedTreeExpression(null, Nullable.ANYTHING) {

    @Override
    public Object evaluate(Object scope) {
      return null;
    }
  };

  public static ResultOrProblems<RealizedTreeExpression> build(
      ExpressionRealizer realizer,
      AST.ExpressionDecl ast,
      Type inputType
  ) {

    List<Problem> problems = new ArrayList<>();

    if (ast instanceof AST.StructExpression) {
      AST.StructExpression structExpr = (AST.StructExpression) ast;
      List<RealizedTreeExpression> children = new ArrayList<>(structExpr.members.size());
      StructBuilder builder = new StructBuilder(children.size());

      for (AST.ExpressionDecl decl : structExpr.members) {
        ResultOrProblems<RealizedTreeExpression> expr = build(realizer, decl, inputType);

        if (expr.hasErrors()) {
          problems.addAll(expr.getProblems());
        } else {
          children.add(expr.get());
          builder.add(decl.getIdentifier().value, expr.get().getResultType());
        }
      }

      return ResultOrProblems.of(new Structed(structExpr, builder.build(), children), problems);
    } else {
      AST.SimpleExpression simpleExpr = (AST.SimpleExpression) ast;

      ResultOrProblems<RealizedExpression> realized = realizer.realize(inputType, simpleExpr.built);
      if (realized.isPresent()) {
        return ResultOrProblems.of(new Simple(simpleExpr, realized.get()));
      } else {
        return ResultOrProblems.failed(Problems.foundWith(simpleExpr.getIdentifier())
            .withChildren(Problems.foundWith(simpleExpr.built, realized.getProblems())));
      }
    }
  }

  @Getter
  protected final AST.ExpressionDecl expression;

  @Getter
  protected final Type resultType;

  public abstract Object evaluate(Object scope);

  private static class Structed extends RealizedTreeExpression {

    private List<RealizedTreeExpression> children;
    private Struct struct;

    Structed(AST.StructExpression expression, Struct resultType, List<RealizedTreeExpression> children) {
      super(expression, resultType);
      this.children = children;
      this.struct = resultType;
    }

    @Override
    public Object evaluate(Object scope) {

      Tuple tuple = new Tuple(struct);
      int i = 0;
      for (RealizedTreeExpression child : children) {
        tuple.set(i++, child.evaluate(scope));
      }

      return tuple;
    }

    @Override
    public String toString() {
      return String.format("Structed(%s)", children);
    }
  }

  private static class Simple extends RealizedTreeExpression {
    Simple(AST.SimpleExpression expression, RealizedExpression realized) {
      super(expression, realized.getResultType());
      this.realizedExpression = realized;
    }

    private final RealizedExpression realizedExpression;

    @Override
    public Object evaluate(Object scope) {
      return realizedExpression.evaluate(scope);
    }

    @Override
    public String toString() {
      return String.format("Simple(%s)", expression);
    }
  }

}
