/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import static nz.org.riskscape.defaults.classifier.ProblemCodes.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;

import nz.org.riskscape.defaults.classifier.AST.ExpressionDecl;
import nz.org.riskscape.dsl.Lexer;
import nz.org.riskscape.dsl.Token;
import nz.org.riskscape.problem.Problem;


public class ClassifierFunctionParser {

  /**
   * Enum defining the identifiers that are allowed. Also supports alternate lables.
   */
  private enum IDType {

    ID("id"),
    DESCRIPTION("description"),
    CATEGORY("category"),
    RETURN_TYPE("return-type"),
    ARGUMENT_TYPES("argument-types"),
    BEFORE("before", "pre"),
    AFTER("after", "post"),
    DEFAULT("default"),
    DO("do", "function"),
    WHEN("when", "filter"),
    UNKNOWN(null);

    static IDType of(String label) {
      for (IDType id : IDType.values()) {
        if (label.equals(id.label) || label.equals(id.alternate)) {
          return id;
        }
      }
      // unknown is returned if nothing matches. Allow switch statements to work which would fail
      // on nulls.
      return UNKNOWN;
    }

    private final String label;
    private final String alternate;

    IDType(String label) {
      this.label = label;
      this.alternate = null;
    }

    IDType(String label, String alternate) {
      this.label = label;
      this.alternate = alternate;
    }

    String getLabels() {
      if (alternate != null) {
        return String.format("%s or %s", label, alternate);
      }
      return label;
    }
  }

  private static final AST.StructType NO_ARGS = new AST.StructType(
      Token.token(TokenTypes.EOF, ""),
      Collections.emptyList()
  );

  Lexer<TokenTypes> lexer;
  public AST.FunctionDecl parse(String source) {
    lexer = new Lexer<>(TokenTypes.tokens(), source);

    return parseFunctionDecl();
  }

  @SuppressWarnings("unchecked")
  private AST.FunctionDecl parseFunctionDecl() {

    HashMap<IDType, Object> memo = new HashMap<>();
    Token start = null;

    while (!lexer.isEOF()) {
      Token identifier = lexer.expect(TokenTypes.IDENTIFIER);
      if (start == null) {
        start = identifier;
      }
      Object parsed = null;

      IDType ident = IDType.of(identifier.value);
      switch (ident) {
      case ID:
      case DESCRIPTION:
      case CATEGORY:
        parsed = parseMetadata(identifier);
        break;
      case RETURN_TYPE:
      case ARGUMENT_TYPES:
        parsed = parseTypeDecl(identifier);
        break;
      case BEFORE:
      case AFTER:
      case DEFAULT:
      case DO:
        parsed = parseExpression(identifier);
        break;
      case WHEN:
        List<AST.Filter> filters = (List<AST.Filter>)memo.getOrDefault(ident, new ArrayList<>());
        filters.add(parseTree(identifier));
        parsed = filters;
        break;
      default:
        throw new UnexpectedIdentifierException(identifier, IDType.ARGUMENT_TYPES.getLabels(),
            IDType.ID.getLabels(), IDType.DESCRIPTION.getLabels(), IDType.CATEGORY.getLabels(),
            IDType.RETURN_TYPE.getLabels(), IDType.BEFORE.getLabels(), IDType.AFTER.getLabels(),
            IDType.DEFAULT.getLabels(), IDType.DO.getLabels(), IDType.WHEN.getLabels());
      }

      if (memo.put(ident, parsed) != null && ident != IDType.WHEN) {
        // redefining items is not allowed except for the 'when' type, which can continue
        // to add more items to an existing definition
        throw new InvalidClassifierSyntaxException(
           Problem.error(REDEFINITION, identifier.value));
      }
    }

    if (memo.containsKey(IDType.DEFAULT) && memo.containsKey(IDType.DO)) {
       // This is a problem as the 'do' will be ignored because of the 'default'.
       ExpressionDecl functionDecl = (ExpressionDecl)memo.get(IDType.DO);
       ExpressionDecl defaultDecl = (ExpressionDecl)memo.get(IDType.DEFAULT);

       throw new InvalidClassifierSyntaxException(
           Problem.error(FUNCTION_AND_DEFAULT,
               functionDecl.getIdentifier().getLocation().getLine(),
               defaultDecl.getIdentifier().getLocation().getLine()));
    }

    // allow do/function instead of default
    memo.putIfAbsent(IDType.DEFAULT, memo.get(IDType.DO));

    return new AST.FunctionDecl(
        start,
        Optional.ofNullable((AST.Metadata) memo.get(IDType.ID)),
        Optional.ofNullable((AST.Metadata) memo.get(IDType.DESCRIPTION)),
        Optional.ofNullable((AST.Metadata) memo.get(IDType.CATEGORY)),
        (AST.StructType) memo.getOrDefault(IDType.ARGUMENT_TYPES, NO_ARGS),
        Optional.ofNullable((AST.TypeDecl) memo.get(IDType.RETURN_TYPE)),
        Optional.ofNullable((AST.ExpressionDecl) memo.get(IDType.BEFORE)),
        (List<AST.Filter>)memo.getOrDefault(IDType.WHEN, new ArrayList<>()),
        Optional.ofNullable((AST.ExpressionDecl) memo.get(IDType.DEFAULT)),
        Optional.ofNullable((AST.ExpressionDecl) memo.get(IDType.AFTER))
    );

  }

  private AST.TypeDecl parseTypeDecl(Token identifier) {

    Token trailing = parseTrailingIfPresent(identifier);

    if (trailing != null) {
      return new AST.SimpleType(identifier, trailing);
    } else {
      ArrayList<AST.TypeDecl> children = new ArrayList<>();
      while (!lexer.isEOF()) {
        Token token = lexer.peek();

        if (token.type == TokenTypes.IDENTIFIER && token.moreIndented(identifier)) {
          children.add(parseTypeDecl(lexer.next()));
        } else {
          break;
        }
      }

      return new AST.StructType(identifier, children);
    }
  }

  private boolean isChild(Token parent, Token compareTo) {
    return compareTo.moreIndented(parent);
  }


  private AST.Filter parseTree(Token identifier) {
    Token filterExpr = expectOnSameLine(identifier, TokenTypes.EXPRESSION);

    AST.ExpressionDecl orElse = null;
    ArrayList<AST.Filter> children = new ArrayList<>();

    while (!lexer.isEOF()) {
      Token nextIdent = lexer.expect(TokenTypes.IDENTIFIER);
      if (!nextIdent.moreIndented(identifier)) {
        lexer.rewind(nextIdent);
        break;
      }

      IDType ident = IDType.of(nextIdent.value);
      switch (ident) {
      case WHEN:
        children.add(parseTree(nextIdent));
        break;
      case DEFAULT:
      case DO:
        orElse = parseExpression(nextIdent);
        break;
      default:
        // TODO proper parse exception, please
        throw new UnexpectedIdentifierException(nextIdent,
            IDType.WHEN.getLabels(), IDType.DEFAULT.getLabels(), IDType.DO.getLabels());
      }
    }

    return new AST.Filter(identifier, filterExpr, children, Optional.ofNullable(orElse));
  }

  private AST.ExpressionDecl parseExpression(Token identifier) {
    IDType identifierType = IDType.of(identifier.value);
    Token trailing = parseTrailingIfPresent(identifier);

    if (trailing != null) {
      return new AST.SimpleExpression(identifier, trailing);
    } else {
      List<Token> trailingTokens = parseTrailingWithQuotedIdentifiersIfPresent(identifier);
      if (! trailingTokens.isEmpty()) {
        return new AST.SimpleExpression(identifier, trailingTokens);
      }
      ArrayList<AST.ExpressionDecl> children = new ArrayList<>();
      while (!lexer.isEOF()) {
        Token childIdentifier = lexer.peek();
        if (!isChild(identifier, childIdentifier)) {
          break;
        } else {
          lexer.expect(TokenTypes.IDENTIFIER);
        }
        AST.ExpressionDecl childExpression = parseExpression(childIdentifier);
        IDType childIdentifierType = IDType.of(childIdentifier.value);

        // special case - if the child identifier is called 'do' and the parent 'default', we consume this
        // token and assume it has a child - this is because there's a weird inconsistency in the syntax for the default
        // block that applies when no filter matches that sort of needlessly puts another function in there
        if (identifierType.equals(IDType.DEFAULT) && childIdentifierType.equals(IDType.DO)) {
          if (childExpression instanceof AST.SimpleExpression) {
            return new AST.SimpleExpression(identifier, ((AST.SimpleExpression) childExpression).getExpressionParts());
          } else {
            return new AST.StructExpression(identifier, ((AST.StructExpression) childExpression).members);
          }
        } else {
          children.add(childExpression);
        }
      }

      return new AST.StructExpression(identifier, children);
    }
  }

  private AST.Metadata parseMetadata(Token identifier) {
    //A function id may contain : characters which would by seen as an identifier, thus need quoting.
    // Eg "kaiju:stomp". Descriptions may also need to be quoted for similar reasons
    Token value = expectOnSameLine(identifier, TokenTypes.EXPRESSION, TokenTypes.QUOTED_IDENTIFIER);
    return new AST.Metadata(identifier, value);
  }

  private Token expectOnSameLine(Token anchor, TokenTypes... expected) {
    Token matched = lexer.expect(expected);

    if (matched.getLocation().getLine() != anchor.getLocation().getLine()) {
      throw UnexpectedWhitespaceException.notOnSameLine(anchor, matched);
    }

    return matched;
  }

  private Token parseTrailingIfPresent(Token ident) {
    if (lexer.peekType() == TokenTypes.EXPRESSION) {
      Token trailing = lexer.next();
      if (trailing.getLocation().getLine() != ident.getLocation().getLine()) {
        throw UnexpectedWhitespaceException.notOnSameLine(ident, trailing);
      } else {
        return trailing;
      }
    } else {
      return null;
    }
  }

  private List<Token> parseTrailingWithQuotedIdentifiersIfPresent(Token ident) {
    // calling a function with a funky id may require it to be quoted, so we might need to parse something like:
    // function: "my-funky-function"(arg1, arg2)
    //
    // so we peek for quoted identifier, adding it and what ever trails if found.
    // Note we only need to worry about the leading quoted identifier, because any others will get consumed
    // by the trailing expression.
    List<Token> found = new ArrayList<>();
    if (lexer.peekType() == TokenTypes.QUOTED_IDENTIFIER) {
      found.add(lexer.next());
      Token trailing = parseTrailingIfPresent(ident);
      if (trailing != null) {
        found.add(trailing);
      }
    }
    return found;
  }
}
