/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.defaults.classifier;

import static nz.org.riskscape.defaults.classifier.ProblemCodes.*;
import static nz.org.riskscape.engine.Tuple.CoerceOptions.*;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import nz.org.riskscape.dsl.LexerException;
import nz.org.riskscape.dsl.ParseException;
import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.RiskscapeException;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.function.ArgumentList;
import nz.org.riskscape.engine.function.FunctionArgument;
import nz.org.riskscape.engine.function.IdentifiedFunction;
import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.rl.ExpressionRealizer;
import nz.org.riskscape.engine.rl.RealizableFunction;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Struct.StructBuilder;
import nz.org.riskscape.engine.types.Struct.StructMember;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.typeset.IdentifiedType;
import nz.org.riskscape.engine.typeset.MissingTypeException;
import nz.org.riskscape.engine.typeset.TypeSet;
import nz.org.riskscape.engine.typexp.DefaultTypeBuilder;
import nz.org.riskscape.engine.typexp.TypeBuilder;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ExpressionParser;
import nz.org.riskscape.rl.ast.FunctionCall;

@RequiredArgsConstructor
public class ClassifierFunction implements RiskscapeFunction, RealizableFunction {

  private static class Compiled {

    public Struct argumentsStructType;
    public RealizedTreeExpression pre;
    public ReturnTypeInferer body;
    public RealizedTreeExpression post;
    public Type bodyType;
    public Type declaredReturnType;
    private Struct bodyInputType;
    private Struct postInputType;
    private TypeSet typeSet;

    public ResultOrProblems<Struct> bodyInputType() {
      if (bodyInputType == null) {
        if (pre == null) {
          bodyInputType = argumentsStructType;
        } else {
          Type preType = pre.getResultType();
          if (preType instanceof Struct) {
            Struct preStruct = (Struct) preType;
            // Fixme - resultorproblem here
            bodyInputType = argumentsStructType.and(preStruct);
          } else {
            return ResultOrProblems.failed(Problem.error(PRE_NOT_STRUCT, preType));
          }
        }
      }
      return ResultOrProblems.of(bodyInputType);
    }

    public ResultOrProblems<Struct> postInputType() {
      if (postInputType == null) {
        if (! bodyType.findAllowNull(Struct.class).isPresent()) {
          return ResultOrProblems.failed(Problem.error(BODY_NOT_STRUCT, bodyType));
        }
        ResultOrProblems<Struct> bit = bodyInputType().map(bis -> bis.and(bodyType.asStruct()));
        if (bit.hasErrors()) {
          return bit;
        }
        postInputType = bit.get();
      }

      return ResultOrProblems.of(postInputType);
    }

    public Type inferredReturnType() {
      Type returnType;
      if (post == null) {
        returnType = body.isDefaultPresent() ? bodyType : Nullable.of(bodyType);
      } else {
        returnType = post.getResultType();
      }
      if (returnType.findAllowNull(Struct.class).isPresent()) {
        // if the last section (body or post) returns a struct then the returnType should include attributes set
        // from preceding sections.
        return inferredReturnTypeAllAttributes();
      }
      return returnType;
    }

    /**
     * Infers a return type that contains all the attributes that have been set in all sections,
     * pre, body and post.
     * @return
     */
    private Type inferredReturnTypeAllAttributes() {
      if (pre == null && post == null) {
        return bodyType;
      }
      Type produced = pre != null ? pre.getResultType().asStruct() : null;
      produced = mergeToReturnType(produced, bodyType.asStruct());
      if (post != null) {
        produced = mergeToReturnType(produced, post.getResultType().asStruct());
      }

      return produced;
    }

    private Type mergeToReturnType(Type soFar, Type toAdd) {
      if (soFar == null) {
        return toAdd;
      }

      boolean nullSoFar = Nullable.is(soFar);
      Struct soFarStruct = soFar.findAllowNull(Struct.class).get();

      boolean nullableToAdd = Nullable.is(toAdd);
      Struct toAddStruct = toAdd.findAllowNull(Struct.class).get();

      // if soFar was not nullable but toAdd is, then any attributes added from toAdd should be
      // made nullable.
      boolean addNullableEntries = nullableToAdd && ! nullSoFar;

      for (StructMember member: toAddStruct.getMembers()) {
        Type toAddMemberType = Nullable.ifTrue(addNullableEntries, member.getType());
        if (! soFarStruct.hasMember(member.getKey())) {
          soFarStruct = soFarStruct.and(member.getKey(), toAddMemberType);

        } else if (! toAddMemberType.equals(soFarStruct.getEntry(member.getKey()).getType())) {

          Type common = typeSet.computeAncestorNoConversion(
            toAddMemberType,
            soFarStruct.getEntry(member.getKey()).getType()
          );

          soFarStruct = soFarStruct.replace(member.getKey(), common);
        }
      }
      // If both were nullable then we make the result nullable to
      return Nullable.ifTrue(nullSoFar && nullableToAdd, soFarStruct);
    }
  }

  public static ResultOrProblems<ClassifierFunction> build(
      Project project,
      String source
    ) {
    ClassifierFunctionParser functionParser = new ClassifierFunctionParser();


    AST.FunctionDecl parsed;
    try {
      parsed = functionParser.parse(source);
    } catch (LexerException | ParseException ex) {
      return ResultOrProblems.error(ex);
    }

    return ResultOrProblems.of(new ClassifierFunction(parsed, project));
  }

  @Getter
  private final AST.FunctionDecl ast;

  @Getter
  private final Project project;

  private ArgumentList arguments;

  @Override
  public Object call(List<Object> args) {
    throw new UnsupportedOperationException();
  }

  public ResultOrProblems<IdentifiedFunction> identified(Resource resource) {
    String id;
    if (ast.id.isPresent()) {
      id = ast.id.get().value();
    } else {
      return ResultOrProblems.failed(Problem.error(MISSING_ID));
    }

    Category category = Category.UNASSIGNED;
    if (ast.category.isPresent()) {
      String declared = ast.category.get().value();
      try {
        category = Category.valueOf(declared.toUpperCase());
      } catch (IllegalArgumentException ex) {
        return ResultOrProblems.failed(GeneralProblems.notAnOption(declared, Category.class));
      }
    }

    return ResultOrProblems.of(this.identified(
        id,
        ast.description.map(AST.Metadata::value).orElse(null),
        resource.getLocation(),
        category));
  }

  private class Realized implements RiskscapeFunction {

    private final Compiled typeInfo;

    @Getter
    private final ArgumentList arguments;

    @Getter
    private final List<Type> argumentTypes;

    @Getter
    private final Type returnType;

    private final Type inferredReturnType;

    /**
     * When true outputs from each section are added to (or modify) the outputs of preceeding sections
     * to produce the result.
     */
    private final boolean combineAll;
    private final boolean coerceRequired;

    Realized(Compiled typeInfo) {
      this.typeInfo = typeInfo;
      this.arguments = ArgumentList.fromStruct(typeInfo.argumentsStructType);
      this.argumentTypes = getTypesFromArguments();
      this.coerceRequired = typeInfo.declaredReturnType != null;

      this.inferredReturnType = typeInfo.inferredReturnType();
      this.returnType = Nullable.ifTrue(Nullable.is(inferredReturnType),
          typeInfo.declaredReturnType != null ? typeInfo.declaredReturnType : inferredReturnType);

      // If the inferred return type is a struct we need to combine attributes from each part of the
      // tree to build up the result.
      // But if the last part of the tree returns a simple type (maybe an integer) then only that is returned.
      combineAll = inferredReturnType.findAllowNull(Struct.class).isPresent();
    }

    @Override
    public Object call(List<Object> args) {

      Tuple resultTuple = null;

      Tuple input = Tuple.ofValues(typeInfo.argumentsStructType, args.toArray());
      Object lastResult = input;
      if (typeInfo.pre != null) {
        lastResult = typeInfo.pre.evaluate(input);

        if (!typeInfo.pre.resultType.internalType().isInstance(lastResult)) {
          throw new RiskscapeException(String.format(
              "Unexpected java type returned from %s pre.  Was %s, expected %s",
              this,
              lastResult == null ? null : lastResult.getClass(),
              typeInfo.pre.resultType.internalType()
          ));
        }
        Tuple preResult = (Tuple) lastResult;
        Tuple newTuple = new Tuple(typeInfo.bodyInputType().get());
        newTuple.setAll(input);
        newTuple.setAll(input.size(), preResult);
        // TODO memo bodyinput type
        input = newTuple;

        if (combineAll) {
          resultTuple = Tuple.of(Nullable.strip(inferredReturnType).asStruct());
          resultTuple.setAll(preResult);
        }
      }

      lastResult = typeInfo.body.evaluate(input);
      if (lastResult instanceof Tuple) {
        // If last result is a Tuple it is possible that it is:
        // - a subset of all possible attributes
        // - attributes that are Tuples may not be owned by the expected Struct (eg Tuples created by `{a: 10}`}
        // So we coerce to the required body type, as that is recursive, just in case.
        Tuple last = Tuple.coerce(typeInfo.bodyType.findAllowNull(Struct.class).get(),
            ((Tuple) lastResult).toMap(),
            EnumSet.of(MISSING_IGNORED));

        if (combineAll) {
          if (resultTuple == null) {
            resultTuple = Tuple.of(Nullable.strip(inferredReturnType).asStruct());
          }
          for (StructMember member: last.getStruct().getMembers()) {
            Object o = last.fetch(member);
            if (o != null) {
              // don't copy null values, they didn't get set (did they)
              resultTuple.set(member.getKey(), o);
            }
          }
        }

        lastResult = last;
      }

      if (lastResult == null) {
        if (!Nullable.is(typeInfo.bodyType)             //not nullable
            && typeInfo.body.isDefaultPresent()) {      //and has default
          throw new RiskscapeException(String.format(
            "Unexpected null returned from %s body.  Expected %s",
            this,
            typeInfo.bodyType.internalType()
        ));
        }
      } else if (!typeInfo.bodyType.internalType().isInstance(lastResult)) {
        throw new RiskscapeException(String.format(
            "Unexpected java type returned from %s body.  Was %s, expected %s",
            this,
            lastResult == null ? null : lastResult.getClass(),
            typeInfo.bodyType.internalType()
        ));
      }

      if (typeInfo.post != null) {
        Struct bodyType = typeInfo.bodyType.findAllowNull(Struct.class).get();
        Tuple lastTuple = (Tuple) lastResult;
        Object[] newValues = new Object[input.size() + bodyType.size()];
        System.arraycopy(input.toArray(), 0, newValues, 0, input.size());

        if (lastTuple != null) {
          System.arraycopy(lastTuple.toArray(), 0, newValues, input.size(), lastTuple.size());
        }
        input = Tuple.ofValues(typeInfo.postInputType().get(), newValues);

        lastResult = typeInfo.post.evaluate(input);

        if (!typeInfo.post.resultType.internalType().isInstance(lastResult)) {
          throw new RiskscapeException(String.format(
              "Unexpected java type returned from %s post.  Was %s, expected %s",
              this,
              lastResult == null ? null : lastResult.getClass(),
              typeInfo.post.resultType.internalType()
          ));
        }

        if (combineAll) {
          if (resultTuple == null) {
            resultTuple = Tuple.of(Nullable.strip(inferredReturnType).asStruct());
          }
          Tuple last = (Tuple)lastResult;
          for (StructMember member: last.getStruct().getMembers()) {
            resultTuple.set(member.getKey(), last.fetch(member));
          }
        }
      }

      // if combineAll is set we need to return the combined resultTuple
      // else only the lastResult is returned
      Object toReturn = combineAll ? resultTuple : lastResult;

      if (coerceRequired) {
        // but it may need to be coerced to the returnType to remove excess attributes
        return returnType.coerce(toReturn);
      }
      return toReturn;
    }
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(
      RealizationContext context,
      FunctionCall functionCall,
      List<Type> argumentTypes
  ) {
     if (argumentTypes.size() > 0) {
      // Classifier functions need to check that they are being realized with the argument types that
      // they require. Otherwise unexpected behaviour could result if function expects a wrapping type
      // but a base type is provided.
      //
      // This step is skipped when argumentTypes.size == 0, which only occurs when called from validate
      // with null args.
      List<Problem> argProblems = new ArrayList<>(argumentTypes.size());
      TypeSet typeSet = context.getProject().getTypeSet();
      for (int i = 0; i < getArguments().size(); i++) {
        FunctionArgument arg = getArguments().get(i);
        Type requiredType = arg.getType();
        Type givenType = argumentTypes.get(i);

        if (!typeSet.isAssignable(givenType, requiredType)) {
          argProblems.add(ArgsProblems.mismatch(arg, givenType));
        }
      }
      if (!argProblems.isEmpty()) {
        return ResultOrProblems.failed(argProblems);
      }
    }

    ExpressionParser exprParser = new ExpressionParser();
    TypeBuilder builder = new DefaultTypeBuilder(project.getTypeSet());
    ExpressionRealizer realizer = context.getExpressionRealizer();

    List<Problem> all = new ArrayList<>();
    all.addAll(ast.parseExpressions(exprParser));
    all.addAll(ast.parseTypes(builder));

    if (Problem.hasErrors(all)) {
      return ResultOrProblems.failed(all);
    }

    Compiled compiled = new Compiled();
    compiled.typeSet = context.getTypeSet();
    compiled.argumentsStructType = ast.argumentTypesDecl.built;

    if (argumentTypes.size() > 0) {
      //The classifier function needs to be realized with the actual argumentTypes. These should always
      //be equivilant to those contained in ast.argumentTypesDecl.built but will only be the same if
      //they were lookup(type) to match the input data source
      //
      //This step is skipped when argumentTypes.size == 0, which only occurs when called from validate
      //with null args.
      Struct asBuilt =  ast.argumentTypesDecl.built;
      StructBuilder actual = Struct.builder();
      for (int i = 0; i < argumentTypes.size(); i++) {
        StructMember m = asBuilt.getMembers().get(i);
        actual.add(m.getKey(), argumentTypes.get(i));
      }
      compiled.argumentsStructType = actual.build();
    }

    if (ast.pre.isPresent()) {
      ResultOrProblems<RealizedTreeExpression> preExprOr = RealizedTreeExpression
        .build(realizer, ast.pre.get(), compiled.argumentsStructType)
        .map(rExpr -> compiled.pre = rExpr);

      if (preExprOr.hasErrors()) {
        return preExprOr.composeFailure(Problems.foundWith(ast.pre.get().getIdentifier()));
      } else {
        compiled.pre = preExprOr.get();
      }
    }

    if (compiled.bodyInputType().hasErrors()) {
      return ResultOrProblems.failed(compiled.bodyInputType().getProblems());
    }

    ResultOrProblems<RealizedTreeFilter> filterOr = RealizedTreeFilter.build(
        realizer,
        ast,
        compiled.bodyInputType().get()
    );

    if (filterOr.hasErrors()) {
      return ResultOrProblems.failed(filterOr.getProblems());
    } else {
      ResultOrProblems<ReturnTypeInferer> bodyAdaptor = ReturnTypeInferer.build(filterOr.get(),
          context.getProject().getTypeSet());
      if (bodyAdaptor.hasErrors()) {
        return ResultOrProblems.failed(bodyAdaptor.getProblems());
      }
      compiled.body = bodyAdaptor.get();
      compiled.bodyType = bodyAdaptor.get().getResultType();
    }

    if (ast.post.isPresent()) {
      if (compiled.postInputType().hasErrors()) {
        return ResultOrProblems.failed(compiled.postInputType().getProblems());
      }

      ResultOrProblems<RealizedTreeExpression> postExprOr = RealizedTreeExpression.
          build(realizer, ast.post.get(), compiled.postInputType().get());

      if (postExprOr.hasErrors()) {
        return ResultOrProblems.failed(postExprOr.getProblems());
      } else {
        compiled.post = postExprOr.get();
      }
    }

    compiled.declaredReturnType = ast.returnTypeDecl.map(rtd -> rtd.getBuilt()).orElse(null);

    return checkReturnType(compiled);
  }

  class ReturnTypeCheck {
    private Type deNulledDeclared;
    private Struct declaredStruct;

    private Type deNulledInferredType;
    private Struct inferredStruct;
    private Type inferred;
    private Type declared;
    private final AST.TypeDecl returnTypeDecl;

    ReturnTypeCheck(Type inferred, Type declared) {
      this.declared = declared;
      this.deNulledDeclared = Nullable.unwrap(declared);
      this.declaredStruct = deNulledDeclared.find(Struct.class).orElse(null);

      this.inferred = inferred;
      this.deNulledInferredType = Nullable.unwrap(inferred);
      this.inferredStruct = deNulledInferredType.find(Struct.class).orElse(null);
      this.returnTypeDecl = ast.returnTypeDecl.get();
    }

    public ResultOrProblems<RiskscapeFunction> check(Compiled compiled) {

      // special case - single member struct is allowed and will magically work because of coercion support
      if (inferredStruct == null && declaredStruct != null && declaredStruct.getMembers().size() == 1) {
        inferredStruct = Struct.of(declaredStruct.getMemberKeys().get(0), deNulledInferredType);
      }

      // todo - we might want to adjust the return type if inferred is nullable and declared isn't?
      if (inferredStruct == null ^ declaredStruct == null) {

        return ResultOrProblems.failed(Problems.get(TypeProblems.class).mismatch(
            returnTypeDecl.getIdentifier(),
            declared,
            inferred));
      } else {
        if (inferredStruct == null) {
          Type inferredUnwrapped = deNulledInferredType.getUnwrappedType();
          Type declaredUnwrapped = deNulledDeclared.getUnwrappedType();

          // compare types based on the unwrapped type
          // TODO this feels a little... contrived?
          if (!inferredUnwrapped.equals(declaredUnwrapped)) {
            return ResultOrProblems.failed(Problems.get(TypeProblems.class).mismatch(
                returnTypeDecl.getIdentifier(), inferredUnwrapped, declaredUnwrapped));
          } else {
            return ResultOrProblems.of(new Realized(compiled));
          }
        } else {
          return checkStructReturnType(compiled).composeProblems(
              (s, l) -> Problems.foundWith(returnTypeDecl.getIdentifier(), l));
        }
      }
    }

    private ResultOrProblems<RiskscapeFunction> returnTypeMismatch(Problem problem) {
      return ResultOrProblems.failed(Problem.error(RETURN_TYPE_MISMATCH).withChildren(problem));
    }

    // compare each member's invariance
    private ResultOrProblems<RiskscapeFunction> checkStructReturnType(Compiled compiled) {
      for (StructMember declaredMember : declaredStruct.getMembers()) {
        StructMember inferredMember = inferredStruct.getMember(declaredMember.getKey()).orElse(null);
        if (inferredMember == null) {
          if (!Nullable.is(declaredMember.getType())) {
            return returnTypeMismatch(
                Problems.get(TypeProblems.class).structMemberNotProvided(declaredMember, inferredStruct));
          }
        } else {
          if (Nullable.is(inferredMember.getType()) && !Nullable.is(declaredMember.getType())) {
            return returnTypeMismatch(
                Problems.get(TypeProblems.class).mismatch(
                  declaredMember,
                  declaredMember.getType(),
                  inferredMember.getType()
                )
            );
          } else {
            Type unwrappedDeclaredMemberType = Nullable.unwrap(declaredMember.getType());
            Type unwrappedInferredMemberType = Nullable.unwrap(inferredMember.getType());

            if (!unwrappedDeclaredMemberType.equals(unwrappedInferredMemberType)) {
              return returnTypeMismatch(
                  Problems.get(TypeProblems.class).mismatch(
                    declaredMember,
                    unwrappedDeclaredMemberType,
                    unwrappedInferredMemberType
                  )
              );
            }
          }
        }
      }

      return ResultOrProblems.of(new Realized(compiled));
    }

  }

  private ResultOrProblems<RiskscapeFunction> checkReturnType(Compiled compiled) {
    Type declared = compiled.declaredReturnType;
    Type inferred = compiled.inferredReturnType();

    if (declared == null) {
      return ResultOrProblems.of(new Realized(compiled));
    } else {
      return new ReturnTypeCheck(inferred, declared).check(compiled);
    }
  }

  @Override
  public Type getReturnType() {
    ast.returnTypeDecl.ifPresent(td -> td.build(new ArrayList<>(), project.getTypeBuilder()));
    return ast.returnTypeDecl.flatMap(decl -> Optional.ofNullable(decl.getBuilt())).orElse(Types.ANYTHING);
  }

  @Override
  public ArgumentList getArguments() {
    if (arguments == null) {

      TypeBuilder builder = project.getTypeBuilder();
      List<Problem> problems = new ArrayList<>();
      List<FunctionArgument> args = new ArrayList<>(ast.argumentTypesDecl.children.size());
      for (AST.TypeDecl decl : ast.argumentTypesDecl.children) {
        decl.build(problems, builder);
        Type type;
        if (decl.getBuilt() != null) {
          type = decl.getBuilt();
        } else {
          type = Types.ANYTHING;
        }
        args.add(new FunctionArgument(decl.getIdentifier().value, type));
      }
      arguments = new ArgumentList(args);
    }

    return arguments;
  }

  @Override
  public List<Type> getArgumentTypes() {
    return getArguments().getArgumentTypes();
  }

  @Override
  public ResultOrProblems<Boolean> validate(RealizationContext context) {
    List<Problem> problems = new ArrayList<>();
    if (ast.returnTypeDecl.isPresent()) {
      validateTypeExists(ast.returnTypeDecl.get().getBuilt(),
          p -> problems.add(p));
    }
    for (Type argumentType: getArgumentTypes()) {
      validateTypeExists(argumentType, p -> problems.add(p));
    }
    if (!problems.isEmpty()) {
      return ResultOrProblems.failed(problems);
    }
    return realize(context, null, Collections.emptyList()).map(f -> true);
  }

  private void validateTypeExists(Type type, Consumer<Problem> problems) {
    if (type instanceof IdentifiedType) {
      IdentifiedType identified = (IdentifiedType) type;
      try {
        identified.getUnderlyingType();
      } catch (MissingTypeException e) {
        problems.accept(e.getProblem());
      }
    }
  }

  @Override
  public String toString() {
    return String.format("Classifier[args=%s, return-type=%s]", getArguments(), getReturnType());
  }

}
