/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */

package nz.org.riskscape.defaults.auth;

import java.net.URI;
import java.util.Arrays;
import java.util.stream.Collectors;

import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;
import nz.org.riskscape.engine.auth.HttpSecret;
import nz.org.riskscape.engine.auth.SecretBuilder;
import nz.org.riskscape.engine.bind.JavaParameterSet;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.problem.ResultOrProblems;

/**
 * A {@link Secret} for koordinates services which require the key secret to be added into the
 * services API path.
 *
 * E.g `https://example.koordinates.com/services;key=<secret>/wfs/...`
 */
@Getter @Setter @ToString(exclude = "apiKey") @EqualsAndHashCode(callSuper = true)
public class KoordinatesKeySecret extends HttpSecret {

  public static final String FRAMEWORK_NAME = "koordinates";

  public static final JavaParameterSet<KoordinatesKeySecret> PARAMETER_SET
      = JavaParameterSet.fromBindingClass(KoordinatesKeySecret.class);

  public static final SecretBuilder BUILDER = new SecretBuilder(
      FRAMEWORK_NAME,
      KoordinatesKeySecret.class,
      PARAMETER_SET,
      (bound) -> ResultOrProblems.of(PARAMETER_SET.bindToObject(bound).getBoundToObject())
  );

  public KoordinatesKeySecret() {
    super(FRAMEWORK_NAME);
  }

  @ParameterField
  private String apiKey;

  @Override
  public boolean matches(URI uri) {
    return super.matches(uri) && isPathAServicesRequest(uri.getPath());
  }

  @Override
  public void apply(Request request) {
    if (isPathAServicesRequest(request.getPath())) {
      // are there other koordinates endpoints that need some other key handling?
      String[] pathComponents = request.getPath().substring(1).split("/");
      // it looks like a koordinates services request, add in the key
      String servicesPart = pathComponents[0];
      pathComponents[0] = String.format("%s;key=%s",
          servicesPart.substring(
              0,
              servicesPart.contains(";") ? servicesPart.indexOf(";") : servicesPart.length()
          ), apiKey
      );
      request.setPath(Arrays.stream(pathComponents).collect(Collectors.joining("/", "/", "")));
    }
  }

  private boolean isPathAServicesRequest(String path) {
    return path.startsWith("/services/") || path.startsWith("/services;");
  }

}
