/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.cpython;

import static nz.org.riskscape.engine.Assert.*;
import static nz.org.riskscape.engine.Matchers.*;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import java.math.BigDecimal;
import java.net.URI;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Date;
import java.util.Optional;

import org.junit.Test;
import org.locationtech.jts.geom.Geometry;

import nz.org.riskscape.engine.ArgsProblems;
import nz.org.riskscape.engine.SRIDSet;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.rl.EvalException;
import nz.org.riskscape.engine.types.CoverageType;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeRegistry;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinSet;

/**
 * Checks can call CPython functions sensibly.
 */
@SuppressWarnings("unchecked")
public class CPythonFunctionTest extends CPythonBaseTest {

  TypeRegistry typeRegistry = project.getTypeSet().getTypeRegistry();

  @Test
  public void canCallAFunctionWithCPythonImport() throws Exception {
    // the equivalent import in Jython would be: from java.util import Random
    String script = ""
        + "import random\n"
        + "def function(a, b):\n"
        + "    return random.uniform(a, b)\n"
        + "\n";

    addFunction("rando", script, Arrays.asList(Types.FLOATING, Types.FLOATING), Types.FLOATING);
    assertThat((Double) evaluate("rando(10, 20)"), allOf(greaterThanOrEqualTo(10D), lessThanOrEqualTo(20D)));
  }

  @Test
  public void inlineFunctionCanImportRelativeToProjectLocation() throws Exception {
    // Functions defined inline are saved to a tmp file which cpython executes
    // and cpython is set to resolve imports relative to the project home
    // And when running on platform the project root is a https resource that is resolved
    // by a mounted platform resource (which will return a getLocal() path)
    //
    // we can test both of these cases by
    // - creating project/function directories (tmp/project/function)
    // - having the inline function:
    //   - saved to a different directory (tmp)
    //   - import a function from a project relative location
    // - setting the project relative to location to one that will the project dir from Resource#getLocal()
    Path projectDir = Files.createDirectories(tmpDir.resolve("project"));
    Path functionDir = Files.createDirectories(projectDir.resolve("functions"));
    writeFile("foo.py", functionDir, """
                                def foo(a, b):
                                  return a + b
                                """);
    // write the script to tmpDir with an import from the local dir
    String script = ""
        + "from functions.foo import foo\n"
        + "def function(a, b):\n"
        + "    return foo(a, b)\n"
        + "\n";

    addFunction("sum", script, Arrays.asList(Types.FLOATING, Types.FLOATING), Types.FLOATING);

    // now we bait and switch the project relative to URI to a custom URI with will map to a project.ini
    // file in the function dir
    customProtocolTestingResourceLoader.addContent("project", "ignored",
        Optional.of(projectDir.resolve("project.ini")));
    project.setRelativeTo(URI.create("test:project"));

    assertThat((Double) evaluate("sum(10, 20)"), is(30D));
  }

  @Test
  public void cannotCallAFunctionWithJythonImport() throws Exception {
    // trying to import a Jython module should throw an import error
    String script = ""
        + "from nz.org.riskscape.engine.types import Types\n"
        + "def function(a):\n"
        + "    return a + 1\n"
        + "\n";

    addFunction("badimport", script, Arrays.asList(Types.FLOATING), Types.FLOATING);
    Exception ex = assertThrows(EvalException.class,
        () -> evaluate("badimport(1)"));
    assertThat(render(ex), allOf(
        containsString("from nz.org.riskscape.engine.types import Types"),
        containsString("ModuleNotFoundError: No module named 'nz'")
    ));
  }

  @Test
  public void canCallAFunctionWithMainStatement() throws Exception {
    // standard case where user wants to debug their own function
    String script = ""
        + "def function(a):\n"
        + "    return 'cool ' + str(a)\n"
        + "\n"
        + "if __name__ == '__main__':\n"
        + "    print(function('test'))\n"
        + "\n";

    addFunction("foo", script, Arrays.asList(Types.INTEGER), Types.TEXT);
    assertThat(evaluate("foo(1)"), is("cool 1"));
  }

  @Test
  public void canCallAFunctionWithKeywordArgs() throws Exception {
    // this would save a user having two variants of their python loss function - one that
    // takes a resource and one that doesn't. (They'd still need 2 INI file definitions)
    String script = ""
        + "def function(a, resource=None):\n"
        + "    b = 0 if resource is None else resource\n"
        + "    return a + b\n"
        + "\n";

    // call with just a single args
    addFunction("foo", script, Arrays.asList(Types.INTEGER), Types.FLOATING);
    assertThat(evaluate("foo(1)"), is(1D));

    // add 2nd function that takes a 'resource' arg
    addFunction("foo_with_resource", script, Arrays.asList(Types.INTEGER, Types.INTEGER), Types.FLOATING);
    assertThat(evaluate("foo_with_resource(1, 2)"), is(3D));
  }

  @Test
  public void canCallaPythonFunctionWithVariableArgs() throws Exception {
    // this is mostly just to check it doesn't die horribly, rather than proving
    // this is a use-case of any value
    String script = ""
        + "def function(*args):\n"
        + "    return sum(args)\n"
        + "\n";

    addFunction("foo", script, Arrays.asList(Types.INTEGER, Types.INTEGER), Types.FLOATING);

    assertThat(evaluate("foo(1, 2)"), is(3D));
  }

  @Test
  public void cannotBuildACPythonFunctionWithNoArgs() throws Exception {
    // no args doesn't work nicely with serialization currently and there doesn't
    // seem to be a compelling reason to make it work
    String script = ""
        + "def function():\n"
        + "    return 'baloney'\n"
        + "\n";

    assertThat(
        makeFunctionOr(script, Arrays.asList(), Types.TEXT),
        failedResult(is(CPythonFunctionFramework.PROBLEMS.zeroArgsNotSupported())
    ));
  }

  @Test
  public void canCallAFunctionThatAcceptsAnything() throws Exception {
    Geometry testGeom = project.getSridSet().getGeometryFactory(SRIDSet.EPSG2193_NZTM).createPoint();
    // geometry gets passed to cpython as wkb (that's bytes)
    String testGeomExpectedWkbString = "(b'\\x00\\x00\\x00\\x00\\x01\\x7f\\xf8\\x00\\x00\\x00\\x00\\x00"
        + "\\x00\\x7f\\xf8\\x00\\x00\\x00\\x00\\x00\\x00', 2)";

    String script = ""
        + "def function(arg):\n"
        + "  return arg\n"
        + "\n";

    addFunction("foo", script, Arrays.asList(Types.ANYTHING), Types.TEXT);

    // give it some simple types
    assertThat(evaluate("foo(3)"), is("3"));
    assertThat(evaluate("foo(3.2)"), is("3.2"));
    assertThat(evaluate("foo('bar')"), is("bar"));
    input = Tuple.ofValues(Struct.of("value", Types.GEOMETRY), testGeom);
    assertThat(evaluate("foo(value)"), is(testGeomExpectedWkbString));

    // now give it some struct input
    assertThat(evaluate("foo({bar: 3, baz: 'bazza'})"), is("{'bar': 3, 'baz': 'bazza'}"));

    Struct childType = Struct.of("animal", new WithinSet(Types.TEXT, "cat", "dog"),
        "geom", Referenced.of(Types.GEOMETRY, SRIDSet.EPSG2193_NZTM),
        "name", Types.TEXT
    );

    input = Tuple.ofValues(Struct.of("value", childType), Tuple.ofValues(childType, "cat", testGeom, "Lexi"));
    String expectedResult = String.format("{'animal': 'cat', 'geom': %s, 'name': 'Lexi'}", testGeomExpectedWkbString);
    assertThat(evaluate("foo(value)"), is(expectedResult));
  }

  @Test
  public void cannotCallFunctionsThatAcceptAnythingWithTypesThatDoNotSerialize() throws Exception {
    // just because a function says it can accept anything doesn't mean that RiskScape can deliver
    // anything to it. There are still some RiskScape types that cpython does not have serializers for.
    // Namely date and decimal.
    String script = ""
        + "def function(arg):\n"
        + "  return arg\n"
        + "\n";

    addFunction("foo", script, Arrays.asList(Types.ANYTHING), Types.TEXT);

    // for these tests we expect to catch an EvalException which will contain a RiskscapePyException
    // from serializer.py
    // date
    input = Tuple.ofValues(Struct.of("value", Types.DATE), new Date());
    assertThrows(EvalException.class, () -> evaluate("foo(value)"));

    // decimal
    input = Tuple.ofValues(Struct.of("value", Types.DECIMAL), new BigDecimal("12.111"));
    assertThrows(EvalException.class, () -> evaluate("foo(value)"));
  }

  @Test
  public void cannotBuildACPythonFunctionWithMissingArgumentType() throws Exception {
    addFunction("missing_arg_type", """
                                   argument-types = [lookup('missing')]
                                   source = ```
                                    def function(foo):
                                      return foo
                                    ```
                                   """);
    assertThat(functionProblems, contains(isError(GeneralProblems.class, "failedResourceLookedUp")));
  }

  @Test
  public void cannotBuildACPythonFunctionWithMissingReturnType() throws Exception {
    addFunction("missing_arg_type1", """
                                   return-type = missing
                                   source = ```
                                    def function(foo):
                                      return foo
                                    ```
                                   """);
    assertThat(functionProblems, contains(isError(GeneralProblems.class, "failedResourceLookedUp")));

    addFunction("missing_arg_type2", """
                                   return-type = lookup('missing')
                                   source = ```
                                    def function(foo):
                                      return foo
                                    ```
                                   """);
    assertThat(functionProblems, contains(isError(GeneralProblems.class, "failedResourceLookedUp")));
  }

  @Test
  public void cannotBuildACPythonFunctionThatReturnsAnything() throws Exception {
    String script = ""
        + "def function(arg):\n"
        + "  return arg\n"
        + "\n";

    assertThat(
        makeFunctionOr(script, Arrays.asList(Types.FLOATING), Types.ANYTHING), failedResult(
        is(
            CPythonFunctionFramework.PROBLEMS.typeNotSupported("anything",
                "'return-type'",
                framework.getSupportedReturnTypeNames(typeRegistry))
        )
    ));
  }

  @Test
  public void cannotBuildACPythonFunctionWithBadReturnType() throws Exception {
    String script = ""
        + "def function(x):\n"
        + "    return 'baloney'\n"
        + "\n";

    Type coverage = new CoverageType(Types.TEXT);

    assertThat(
        makeFunctionOr(script, Arrays.asList(Types.FLOATING), coverage),
        failedResult(is(
            CPythonFunctionFramework.PROBLEMS.typeNotSupported("coverage",
                "'return-type'",
                framework.getSupportedReturnTypeNames(typeRegistry)
            )
        ))
    );
  }

  @Test
  public void canRunACPythonFunctionWithPrintStatements() throws Exception {
    // print statements just get swallowed, but they shouldn't mess with serialization
    String script = ""
        + "import sys\n"
        + "def function(x):\n"
        + "    print('first, ')\n"
        + "    print('do no harm', file=sys.stdout)\n"
        + "    sys.stdout.flush()\n"
        + "    return x + ' baz'\n"
        + "\n";

    addFunction("foo", script, Arrays.asList(Types.TEXT), Types.TEXT);
    assertThat(evaluate("foo('bar')"), is("bar baz"));

    assertThat(
      sunkProblems,
      contains(
        CPythonProblems.get().functionOutput(pythonScript, "first,"),
        CPythonProblems.get().functionOutput(pythonScript, "do no harm")
      )
    );
  }

  @Test
  public void canGetCPythonDebugFromStderr() throws Exception {
    // print statements go to stderr and sent to the problem sink, exceptions are caught and serialized
    String script = ""
        + "import sys\n"
        + "def function(x):\n"
        + "    print('I have seen things', file=sys.stderr)\n"
        + "    print('terrible things', file=sys.stderr)\n"
        + "    if x < 10:\n"
        + "        raise RuntimeError('what went wrong?')\n"
        + "    return str(x) + ' is OK'\n"
        + "\n";

    addFunction("foo", script, Arrays.asList(Types.FLOATING), Types.TEXT);

    // stderr doesn't interfere with normal operation of the function
    assertThat(evaluate("foo(11)"), is("11.0 is OK"));

    // NB looks like this test can be flappy - there's a possibility that the stderr output gets missed due to a timing
    // issue, so for this test alone, we do a second check for stderr output to make sure that the reaper does a full
    // check (instead of maybe being called at the very end of the stderr check and missing the output)
    // XXX we might want to make this part of the check logic anyway, so there's more certainty that we'll see
    // the output - not sure, as it's kind of a best effort thing here
    runner.checkForOutputAndWait();

    assertThat(
      sunkProblems,
      contains(
        CPythonProblems.get().functionOutput(pythonScript, "I have seen things"),
        CPythonProblems.get().functionOutput(pythonScript, "terrible things")
      )
    );

    // but it only gets displayed if we get a function call exception
    Exception ex = assertThrows(EvalException.class, () -> evaluate("foo(9)"));
    assertThat(render(ex),
      containsString("RuntimeError: what went wrong?")
    );
  }

  @Test
  public void canRunACPythonFunctionWithSomeNonTrivialArgsGL578() throws Exception {
    // the main reason the cpython function is realizable is to make sure serialize using the types we're given, rather
    // than the types the function was declared with, which was causing struct owner issues
    String script = ""
        + "def function(list):\n"
        + "  return sum([x['val'] for x in list])\n";

    addFunction("sum", script, Arrays.asList(RSList.create(Struct.of("val", Types.INTEGER))), Types.INTEGER);

    assertThat(
      evaluate("sum([{val: 1}, {val: 2}, {val: 3}])"),
      equalTo(6L)
    );
  }

  @Test
  public void canRunACPythonFunctionWithSomeStandardCoercibleArgs() throws Exception {
    // just to make sure we're not interrupting argument coercion
    String script = ""
        + "def function(x):\n"
        + "  return len(x)\n";

    addFunction("foo", script, Arrays.asList(Struct.of("bar", Types.TEXT)), Types.INTEGER);

    assertThat(
      evaluate("foo({bar: 'ok', baz: 'go away'})"),
      equalTo(1L)
    );
  }

  @Test
  public void canRunACPythonFunctionWithSomeSpecificGeometryArgs() throws Exception {
    // just to make sure we're not interrupting argument coercion
    String script = ""
        + "def function(x):\n"
        + "  return type(x)\n";

    Type givenArgType = Referenced.of(Types.POLYGON, SRIDSet.EPSG2193_NZTM);
    Type declaredArgType = Types.POLYGON;


    addFunction("foo", script, Arrays.asList(declaredArgType), Types.TEXT);

    input = Tuple.ofValues(
      givenArgType.asStruct(),
      project.getSridSet().getGeometryFactory(SRIDSet.EPSG2193_NZTM).createPolygon()
    );

    assertThat(
      evaluate("foo(value)"),
      equalTo("<class 'tuple'>")
    );
  }

  @Test
  public void canRunACPythonFunctionWithSomeStandardAssignableArgs() throws Exception {
    // assignable types are not touched by realization, so our function needs to detect them using isAssignable
    String script = ""
        + "def function(x):\n"
        + "  return type(x)\n";

    addFunction("foo", script, Arrays.asList(Types.GEOMETRY), Types.TEXT);

    input = Tuple.ofValues(
      Types.POLYGON.asStruct(),
      project.getSridSet().getGeometryFactory(SRIDSet.EPSG2193_NZTM).createPolygon()
    );

    assertThat(
      evaluate("foo(value)"),
      equalTo("<class 'tuple'>")
    );
  }

  @Test
  public void canRunACPythonFunctionWithLinkedType() throws Exception {
    // assignable types are not touched by realization, so our function needs to detect them using isAssignable
    String script = ""
        + "def function(x):\n"
        + "  return type(x)\n";

    Type fooType = Struct.of("int", Types.INTEGER);
    project.getTypeSet().add("foo", fooType);

    addFunction("foo", script, Arrays.asList(project.getTypeSet().getLinkedType("foo")), Types.TEXT);



    assertThat(
      evaluate("foo({int: 12})"),
      equalTo("<class 'dict'>")
    );
  }

  @Test
  public void willFailSensiblyIfArgsAreNotCoercible() throws Exception {
    // make sure failing argument coercion causes realization to fail
    String script = ""
        + "def function(x):\n"
        + "  return len(x)\n";

    addFunction("foo", script, Arrays.asList(Struct.of("bar", Types.TEXT, "baz", Types.INTEGER)), Types.INTEGER);

    evaluate("foo({bar: 'ok'})", true);
    assertThat(
      realizationProblems,
      contains(
            hasAncestorProblem(
                equalIgnoringChildren(
                    ArgsProblems.mismatch(built.getArguments().get(0), Struct.of("bar", Types.TEXT))
                )
            )
      )
    );
  }
}
