/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.cpython;

import java.util.concurrent.atomic.AtomicBoolean;

import lombok.extern.slf4j.Slf4j;

/**
 *
 */
@Slf4j
class Reaper extends Thread {

  private static final int SLEEP_TIME_MILLIS = 100;

  /**
   * Mutex used for signalling the reaper that it needs to wake up
   */
  private final Object reaperWakeup = new Object();

  /**
   * Flag to tell the reaper thread it's time to stop.
   */
  private final AtomicBoolean stopped = new AtomicBoolean(false);

  private final CPythonSpawner spawner;

  Reaper(CPythonSpawner spawner) {
    super("reaper");
    this.setDaemon(true);
    this.spawner = spawner;
    start();
  }

  /**
   * Loops for ever, and checks each process for:
   *  * Any output from stderr, which gets echoed to riskscape's console via the a problem sink
   *  * Any processes that are dead - they are removed from the list and checked for non-zero exit
   */
  @Override
  public void run() {
    log.info("CPython process reaper thread starting");

    while (true) {
      try {
        if (stopped.get()) {
          log.info("Reaper thread stopping");
          return;
        }

        boolean noOutputCaptured = spawner.checkProcesses();

        // if there was no output from any of the processes, we sleep for a bit to avoid needlessly spin-locking.
        // Wouldn't it be nice if Java supported async io on a process' streams?
        if (noOutputCaptured) {
          synchronized (reaperWakeup) {
            // we wait on the reaperWakeup - if a thread calls `waitForReaper`, we don't want to keep them waiting
            // until we reach the full REAPER_SLEEP_TIME_MILLIS
            reaperWakeup.wait(SLEEP_TIME_MILLIS);
          }
        }

      } catch (Throwable t) {
        log.error("Error during reaper loop - {}", t);
      }
    }
  }


  void stopReaping() {
    stopped.set(true);
    spawner.killAll();

    if (isAlive()) {
      synchronized (reaperWakeup) {
        reaperWakeup.notify();
      }
    }

  }

  void wakeup() {
    synchronized (reaperWakeup) {
      reaperWakeup.notify();
    }
  }

}
