/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.cpython;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.pool2.BasePooledObjectFactory;
import org.apache.commons.pool2.PooledObject;
import org.apache.commons.pool2.impl.DefaultPooledObject;
import org.apache.commons.pool2.impl.GenericObjectPool;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;

import lombok.RequiredArgsConstructor;
import nz.org.riskscape.cpython.CPythonSpawner.CPythonProcess;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.function.FunctionMetadata;
import nz.org.riskscape.engine.function.RiskscapeFunction;
import nz.org.riskscape.engine.function.UserDefinedFunction;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.rl.RealizationContext;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.problem.ResultOrProblems;
import nz.org.riskscape.rl.ast.FunctionCall;

public class CPythonRealizableFunction extends UserDefinedFunction {

  final CPythonSpawner runner;
  private final Project project;

  @RequiredArgsConstructor
  class FunctionFactory extends BasePooledObjectFactory<CPythonProcess> {

    private final List<Type> argumentTypes;

    @Override
    public CPythonProcess create() throws Exception {
      List<String> pyArgs = new ArrayList<>();
      pyArgs.add(CPythonSerializer.serializerType(getMetadata().getReturnType()));
      // we "lie" slightly about the argument type here to guarantee null safety
      argumentTypes.forEach(type -> pyArgs.add(CPythonSerializer.serializerType(CPythonSerializer.nullSafeType(type))));

      CPythonProcess childProcess = runner.startWrapperScript(
          identified(),
          getScript(),
          project,
          pyArgs.toArray(new String[pyArgs.size()]));

      return childProcess;
    }

    @Override
    public void destroyObject(PooledObject<CPythonProcess> p) throws Exception {
      p.getObject().destroy();
    }

    @Override
    public PooledObject<CPythonProcess> wrap(CPythonProcess obj) {
      return new DefaultPooledObject<CPythonProcess>(obj);
    }
  };

  public CPythonRealizableFunction(FunctionMetadata details,
                                   Resource pythonScript,
                                   CPythonSpawner runner,
                                   Project project) {
    super(details, pythonScript);
    this.runner = runner;
    this.project = project;
  }

  @Override
  public ResultOrProblems<RiskscapeFunction> realize(RealizationContext context, FunctionCall functionCall,
      List<Type> argumentTypes) {

    if (areArgumentsCompatible(context, argumentTypes)) {
      GenericObjectPool<CPythonProcess> pool =
          new GenericObjectPool<>(new FunctionFactory(argumentTypes), getPoolConfig());

      return ResultOrProblems.of(new CPythonFunction(argumentTypes, this, pool));
    } else {
      return ResultOrProblems.failed(getArgumentProblems(context, argumentTypes));
    }
  }

  private GenericObjectPoolConfig<CPythonProcess> getPoolConfig() {
    GenericObjectPoolConfig<CPythonProcess> poolConfig = new GenericObjectPoolConfig<>();

    // there's not much point rotating processes - if anything, we want the opposite to be true - keep using the same
    // ones as much as possible
    poolConfig.setLifo(true);
    // being fair sounds like it will waste cpu cycles - we don't care if an object gets hardly used or not
    poolConfig.setFairness(false);

    // no JMX for RiskScape
    poolConfig.setJmxEnabled(false);

    // -1 means no max - we can wait forever
    poolConfig.setMaxWait(Duration.ofMillis(-1));
    // functions will need to block while waiting for a process to become available
    poolConfig.setBlockWhenExhausted(true);

    // we already do this in our creation call anyway as part of booting the cpython process
    // we are not expecting our functions to die behind our back, no point doing a process.isAlive check on every
    // function call
    poolConfig.setTestOnBorrow(false);
    // same here - plus these are happening in the main thread, too
    poolConfig.setTestWhileIdle(false);
    poolConfig.setTestOnCreate(false);
    poolConfig.setTestOnReturn(false);

    // nb this is per function, not per pipeline
    poolConfig.setMaxTotal(runner.getSettings().getMaxProcessesPerFunction());
    // allow the same number idle as allowed at all - we will rely on other mechanisms to clean up after us
    poolConfig.setMaxIdle(runner.getSettings().getMaxProcessesPerFunction());
    // it's desirable to start with no processes until they're needed - a small ramp up cost is fine, not an interactive
    // up
    poolConfig.setMinIdle(0);
    // don't run the evictor
    poolConfig.setTimeBetweenEvictionRuns(Duration.ofMillis(-1));

    return poolConfig;
  }
}
