/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.cpython;

import java.nio.file.Path;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import javax.annotation.Nullable;

import lombok.Getter;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.plugin.Plugin;
import nz.org.riskscape.engine.plugin.PluginDescriptor;
import nz.org.riskscape.engine.plugin.PluginRuntimeException;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemSink;

public class CPythonPlugin extends Plugin {

  @Nullable
  private CPythonSettings settings;

  private boolean checkPassed = false;

  @Getter
  private CPythonSpawner spawner;

  public CPythonPlugin(PluginDescriptor pDescriptor) {
    super(pDescriptor);
  }

  @Override
  public void startUp(Map<String, List<String>> iniSection, ProblemSink problemSink) throws PluginRuntimeException {
    super.startUp(iniSection);

    // TODO maybe we want a method on plugin to make it assert it's installed on a filesystem?
    Path pluginPath = this.getDescriptor().getSourcePath();

    settings = CPythonSettings.fromIniSection(pluginPath, iniSection);

    // only run the check if the user has explicitly set python3-bin
    if (settings.isPython3BinSet()) {
      checkPassed = settings.checkPython3Working(problemSink);
    }
  }

  @Override
  public List<Problem> initializeEngine(Engine engine) {

    // if the python3 check failed or wasn't attempted, we put a 'crippled' function framework in to the engine
    // so that attempts to build cpython functions give a warning "just in time"
    CPythonFunctionFramework functionFramework;

    if (checkPassed) {
      spawner = new CPythonSpawner(settings, engine.getProblemSink());
      functionFramework = new CPythonFunctionFramework(spawner);

      if (engine.isBetaPluginEnabled()) {
        engine.getPipelineSteps().add(new CPythonExecStep(engine, spawner));
      }
    } else {
      functionFramework = CPythonFunctionFramework.notEnabled();
    }
    engine.getFunctionFrameworks().add(functionFramework);

    return Collections.emptyList();
  }

  @Override
  public void shutDown() throws PluginRuntimeException {
    if (spawner != null) {
      spawner.destroy();
    }
  }
}
