/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.cpython;

import java.net.URI;
import java.util.List;
import java.util.Optional;

import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.bind.ParameterField;
import nz.org.riskscape.engine.pipeline.RealizationInput;
import nz.org.riskscape.engine.pipeline.Realized;
import nz.org.riskscape.engine.pipeline.RealizedStep;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.resource.Resource;
import nz.org.riskscape.engine.resource.ResourceLoadingException;
import nz.org.riskscape.engine.steps.BaseStep;
import nz.org.riskscape.engine.steps.Input;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.TypeRegistry;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.Problems;
import nz.org.riskscape.problem.ResultOrProblems;

public class CPythonExecStep extends BaseStep<CPythonExecStep.Parameters> {

  interface LocalProblems extends ProblemFactory {

    // we don't support serializing the given type to CPython
    Problem typeNotSupported(String previousStep, List<String> unsupported, Problem... children);

    Problem typeNotSupportedHint(String previousStep, Struct type);
  }

  static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  private final CPythonSpawner spawner;

  public CPythonExecStep(Engine engine, CPythonSpawner spawner) {
    super(engine);
    this.spawner = spawner;
  }

  @Override
  public String getId() {
    return "python";
  }

  public static class Parameters {
    @Input
    public RealizedStep input;

    public RealizationInput rInput;

    @ParameterField
    public URI script;

    @ParameterField
    public Optional<Struct> resultType;

    @ParameterField
    public Optional<Tuple> parameters;

  }

  @Override
  public ResultOrProblems<? extends Realized> realize(Parameters parameters) {
    Resource scriptResource;
    try {
      scriptResource = getEngine().getResourceFactory().load(parameters.script);
    } catch (ResourceLoadingException ex) {
      return ResultOrProblems.failed(ex.getProblem());
    }

    TypeRegistry registry = parameters.rInput.getExecutionContext().getEngine().getTypeRegistry();
    List<String> unsupported = CPythonFunctionFramework
          .findUnsupportedTypes(registry, parameters.input.getProduces(), CPythonFunction.SUPPORTED_ARGUMENT_TYPES);
    if (!unsupported.isEmpty()) {
      return ResultOrProblems.failed(
          PROBLEMS.typeNotSupported(
              parameters.input.getStepName(),
              unsupported,
              PROBLEMS.typeNotSupportedHint(parameters.input.getStepName(), parameters.input.getProduces())
          )
      );
    }


    CPythonAsyncProjector asyncProjector = new CPythonAsyncProjector(
        parameters.rInput.getStepDefinition(),
        parameters.input.getProduces(),
        parameters.resultType.orElse(Struct.EMPTY_STRUCT),
        parameters.parameters,
        scriptResource,
        spawner
    );

    if (parameters.resultType.isPresent()) {
      return ResultOrProblems.of(asyncProjector);
    } else {
      return ResultOrProblems.of(asyncProjector.getOutput());
    }
  }

}
