/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.beta;

import static nz.org.riskscape.engine.defaults.Plugin.DEFAULTS_SOURCE;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import nz.org.riskscape.config.BootstrapIniSettings;
import nz.org.riskscape.defaults.function.ListToTuple;
import nz.org.riskscape.defaults.function.SwitchFunction;
import nz.org.riskscape.defaults.function.expression.ExpressionFunctionFramework;
import nz.org.riskscape.defaults.interp.CreateBilinearContinuousFunction;
import nz.org.riskscape.engine.DefaultEngine;
import nz.org.riskscape.engine.Engine;
import nz.org.riskscape.engine.Project;
import nz.org.riskscape.engine.defaults.function.LossesByPeriod;
import nz.org.riskscape.engine.function.IdentifiedFunction.Category;
import nz.org.riskscape.engine.function.geometry.ToRelation;
import nz.org.riskscape.engine.function.lang.RemoveAttr;
import nz.org.riskscape.engine.output.GeoTiffFormat;
import nz.org.riskscape.engine.plugin.PluginDescriptor;
import nz.org.riskscape.engine.problem.ProblemFactory;
import nz.org.riskscape.engine.problem.SeverityLevel;
import nz.org.riskscape.engine.rl.LanguageFunctions;
import nz.org.riskscape.engine.steps.SubpipelineStep;
import nz.org.riskscape.postgis.PostGISPipelineOutputStore;
import nz.org.riskscape.problem.Problem;
import nz.org.riskscape.problem.ProblemSink;
import nz.org.riskscape.problem.Problems;

/**
 * A plugin for registering features with the {@link Engine} and/or {@link Project} that are in development.
 *
 * These features could be likely to change, may not be production quality, or may need some real world
 * user testing to inform further development decisions.
 *
 * We don't want RiskScape users inadvertently using these features. They should know what they are up
 * for. Hence this plugin being a home for these features, and there being a warning when this plugin is
 * enabled.
 *
 * Note, the features that are registered by the beta plugin don't have to have their code here too. It makes
 * sense that the code lives in the plugin that will be the features home when in production. This means
 * there is no need to move java files around which makes history tracking a little harder. So write the code
 * in the plugin that should home the feature, but register it here whilst in development.
 */
public class Plugin extends nz.org.riskscape.engine.plugin.Plugin {

  interface LocalProblems extends ProblemFactory {
    @SeverityLevel(Problem.Severity.WARNING)
    Problem betaEnabled();
  }

  static final LocalProblems PROBLEMS = Problems.get(LocalProblems.class);

  public Plugin(PluginDescriptor pDescriptor) {
    super(pDescriptor);

    assert pDescriptor.getPluginId().equals(DefaultEngine.BETA_PLUGIN_ID): "Beta plugin has the wrong id";
  }

  @Override
  public void startUp(Map<String, List<String>> settingsConfig, ProblemSink sink) {
    super.startUp(settingsConfig);
    BootstrapIniSettings settings = new BootstrapIniSettings(settingsConfig, sink);
  }

  @Override
  public List<Problem> initializeEngine(Engine engine) {
    List<Problem> problems = new ArrayList<>();

    // warn the user that the beta plugin is enabled
    problems.add(PROBLEMS.betaEnabled());

    engine.getPipelineOutputStores().add(new PostGISPipelineOutputStore());
    engine.getFunctionFrameworks().add(new ExpressionFunctionFramework());
    engine.getPipelineSteps().add(new SubpipelineStep(engine));

    engine.getFormats().add(new GeoTiffFormat());

    return problems;
  }

  @Override
  public List<Problem> initializeProject(Project project, Engine engine) {
    List<Problem> problems = new ArrayList<>();

    project.getFunctionSet().add(new SwitchFunction().builtin("switch", Category.LANGUAGE));
    project.getFunctionSet().add(new ListToTuple().builtin("list_to_tuple", Category.MISC));

    project.getFunctionSet().add(new LossesByPeriod().builtin("losses_by_period", Category.RISK_METRICS));

    project.getFunctionSet().add(new RemoveAttr().builtin("remove_attr", Category.LANGUAGE), DEFAULTS_SOURCE);
    project.getFunctionSet().add(LanguageFunctions.GET_ATTR);

    project.getFunctionSet().add(new ToRelation()
        .builtin("to_relation", Category.GEOMETRY_PROCESSING), DEFAULTS_SOURCE);

    // continuous functions
    project.getFunctionSet().add(
      new CreateBilinearContinuousFunction().asFunction().builtin("create_continuous_2d", Category.MATHS)
    );

    return problems;
  }

}
