/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.util;

import static nz.org.riskscape.engine.Assert.*;
import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.Mockito.*;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;

import org.junit.Before;
import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.relation.EmptyRelation;
import nz.org.riskscape.engine.relation.ListRelation;
import nz.org.riskscape.engine.relation.Relation;
import nz.org.riskscape.engine.relation.TupleIterator;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.problem.Problem;

public class RelationBackedListTest {

  Struct struct = Struct.of("bar", Types.TEXT);
  ListRelation underlyingRelation = ListRelation.ofTypedValues(struct, "one", "two", "three");
  List<Problem> problems = new ArrayList<>();
  RelationBackedList subject = new RelationBackedList(underlyingRelation, p -> problems.add(p));
  RelationBackedList empty = new RelationBackedList(new EmptyRelation(struct), p -> problems.add(p));
  RelationBackedList emptyButUnknownSize;

  @Before
  public void setup() throws Exception {
    Relation mockedRelation = mock(Relation.class);
    when(mockedRelation.size()).thenReturn(Optional.empty());
    when(mockedRelation.iterator()).thenReturn(TupleIterator.EMPTY);
    emptyButUnknownSize = new RelationBackedList(mockedRelation, p -> problems.add(p));
  }

  @Test
  public void canIterateList() throws Exception {
    Iterator<Tuple> iterator = subject.iterator();

    assertTrue(iterator.hasNext());
    assertThat(iterator.next(), is(Tuple.ofValues(struct, "one")));
    assertTrue(iterator.hasNext());
    assertThat(iterator.next(), is(Tuple.ofValues(struct, "two")));
    assertTrue(iterator.hasNext());
    assertThat(iterator.next(), is(Tuple.ofValues(struct, "three")));
    assertFalse(iterator.hasNext());

    assertFalse(empty.iterator().hasNext());
    assertFalse(emptyButUnknownSize.iterator().hasNext());
    assertThat(problems, empty());
  }

  @Test
  public void canGetListItem() throws Exception {
    assertThat(subject.get(0), is(Tuple.ofValues(struct, "one")));
    assertThat(subject.get(1), is(Tuple.ofValues(struct, "two")));
    assertThat(subject.get(2), is(Tuple.ofValues(struct, "three")));

    assertThrows(IndexOutOfBoundsException.class, () -> subject.get(3));
    assertThrows(IndexOutOfBoundsException.class, () -> subject.get(-1));
    assertThrows(IndexOutOfBoundsException.class, () -> empty.get(0));

    assertThrows(IndexOutOfBoundsException.class, () -> emptyButUnknownSize.get(0));
    assertThat(problems, empty());
  }

  @Test
  public void canGetListSize() throws Exception {
    assertThat(subject.size(), is(3));
    assertFalse(subject.isEmpty());

    assertThat(empty.size(), is(0));
    assertTrue(empty.isEmpty());

    assertThat(emptyButUnknownSize.size(), is(0));
    assertTrue(emptyButUnknownSize.isEmpty());

    assertThat(problems, empty());
  }

  @Test
  public void unsupportedListOperationsAreRejected() throws Exception {
    // NB most of these operations do actually work in a pipeline, because
    // the list functions like concat() actually use ListUtils
    assertThrows(UnsupportedOperationException.class,
        () -> subject.add(Tuple.ofValues(struct, "four")));
    assertThrows(UnsupportedOperationException.class,
        () -> subject.add(0, Tuple.ofValues(struct, "zero")));
    assertThrows(UnsupportedOperationException.class,
        () -> subject.addAll(List.of(Tuple.ofValues(struct, "four"), Tuple.ofValues(struct, "five"))));
    assertThrows(UnsupportedOperationException.class,
        () -> subject.sort((l, r) -> 0));
    assertThat(problems, empty());
  }
}
