/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.typeset;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;

import org.junit.Test;

import nz.org.riskscape.engine.ObjectAlreadyExistsException;
import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.eqrule.Coercer;

public class TypeSetTest {

  TypeSet typeSet = new TypeSet();

  @Test
  public void isEmptyOnConstruction() {
    assertTrue(typeSet.isEmpty());
  }

  @Test
  public void canAddATypeById() {
    CanonicalType type = typeSet.add("foo", Types.ANYTHING);
    assertFalse(typeSet.isEmpty());
    assertNotNull(type);

    assertEquals("foo", type.getId());
    assertSame(typeSet, type.getTypeSet());
    assertSame(Types.ANYTHING, type.getUnderlyingType());
    assertSame(type, typeSet.getRequired("foo"));
  }

  @Test
  public void canAddTheSameUnderlyingTypeManyTimes() {
    CanonicalType fooType = typeSet.add("foo", Types.ANYTHING);
    CanonicalType barType = typeSet.add("bar", Types.ANYTHING);

    assertNotEquals(fooType, barType);
    assertSame(fooType.getUnderlyingType(), barType.getUnderlyingType());
  }

  @Test(expected=ObjectAlreadyExistsException.class)
  public void refusesToAddTheSameIdInTwice() {
    typeSet.add("foo", Types.ANYTHING);
    typeSet.add("foo", Types.ANYTHING);
  }

  @Test
  public void canReturnALinkedType() {
    CanonicalType ct = typeSet.add("foo", Types.ANYTHING);
    LinkedType lt = typeSet.getLinkedType("foo");

    assertEquals(ct.getId(), lt.getId());
    assertSame(ct, lt.getCanonicalType());
    assertSame(ct.getUnderlyingType(), lt.getUnderlyingType());
  }

  @Test
  public void linkedTypesDontThrowOnUseIfTypeIsMissingOnCreation() {
    typeSet.getLinkedType("foo");
  }

  @Test
  public void linkedTypesThrowOnUseIfTypeIsMissingOnUse() {
    LinkedType linkedType = typeSet.getLinkedType("foo");

    expectThrows(MissingTypeException.class, () -> linkedType.getUnderlyingType());
    expectThrows(MissingTypeException.class, () -> linkedType.getCanonicalType());
    expectThrows(MissingTypeException.class, () -> linkedType.coerce("foo"));
  }

  @Test
  public void findsIdentityCoercerForSameTypes() {
    Coercer coercer = typeSet.findEquivalenceCoercer(Types.INTEGER, Types.INTEGER).orElse(null);
    assertThat(coercer.isIdentity(), is(true));
    assertThat(coercer.apply(10L), is(10L));

    Struct type = Struct.of("foo", Types.TEXT);
    coercer = typeSet.findEquivalenceCoercer(type, type).orElse(null);
    assertThat(coercer.isIdentity(), is(true));
    Tuple foo = Tuple.ofValues(type, "bar");
    assertThat(coercer.apply(foo), sameInstance(foo));
  }

  private void expectThrows(Class<? extends Exception> expected, Runnable r) {
    try {
      r.run();
      fail("Expected runnable to throw " + expected);
    } catch(RuntimeException ex) {
      assertEquals(expected, ex.getClass());
    }
  }


}
