/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.eqrule;

import static org.junit.Assert.*;

import org.geotools.referencing.crs.DefaultGeographicCRS;
import org.junit.Test;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.Point;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Referenced;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeRegistry;
import nz.org.riskscape.engine.types.TypeRulesTest;
import nz.org.riskscape.engine.types.Types;

public class StructGeometryRuleTest extends TypeRulesTest {

  GeometryFactory gf = new GeometryFactory();
  Struct onlyGeom = Struct.of("geom", Types.GEOMETRY);
  Struct onlyPointGeom = Struct.of("geom", Types.POINT);
  Struct manyGeom = Struct.of("geom", Types.GEOMETRY, "other_geom", Types.GEOMETRY);
  Struct oneGeom = Struct.of("geom", Types.GEOMETRY, "text", Types.TEXT);
  Struct onePointGeom = Struct.of("geom", Types.POINT, "text", Types.TEXT);
  // geom is not the first member, so it is clear if rule is descending structure
  Struct oneGeomAndChildGeom = Struct.of("child", oneGeom, "geom", Types.GEOMETRY);
  Struct noGeom = Struct.of("foo", Types.TEXT, "geom", Types.INTEGER);
  Struct noGeomButChildDoes = Struct.of("foo", Types.TEXT, "bar", onlyGeom);
  Struct noGeomButNullableChildDoes = Struct.of("foo", Types.TEXT, "bar", Nullable.of(onlyGeom));
  Struct nullableGeom = Struct.of("geom", Nullable.GEOMETRY);

  Point geom = gf.createPoint();
  Type geometry = Types.GEOMETRY;

  @Override
  protected void addTypes() {
    TypeRegistry.addDefaults(registry);
  }

  @Test
  public void aStructWithASingleGeometryMemberIsEquivalentToGeometry() throws Exception {
    coercer = find(onlyGeom, geometry);
    assertNotNull(coercer);
    assertSame(geom, coercer.apply(Tuple.ofValues(onlyGeom, geom)));
  }

  @Test
  public void aStructWithASingleReferencedGeometryMemberIsEquivalentToGeometry() throws Exception {
    Type referencedType = Referenced.of(Types.GEOMETRY, DefaultGeographicCRS.WGS84);
    Struct onlyReferencedGeom = Struct.of("geom", referencedType, "value", Types.INTEGER);

    coercer = find(onlyReferencedGeom, geometry);
    assertNotNull(coercer);
    assertSame(referencedType, coercer.getTargetType());
    assertSame(geom, coercer.apply(Tuple.ofValues(onlyReferencedGeom, geom)));

    // with the current rule ordering this geom only struct will be handled by the single value struct rule
    onlyReferencedGeom = Struct.of("geom", referencedType);

    coercer = find(onlyReferencedGeom, geometry);
    assertNotNull(coercer);
    assertSame(referencedType, coercer.getTargetType());
    assertSame(geom, coercer.apply(Tuple.ofValues(onlyReferencedGeom, geom)));
  }

  @Test
  public void aStructWithASinglePointGeometryMemberIsEquivalentToGeometry() throws Exception {
    coercer = find(onlyPointGeom, geometry);
    assertNotNull(coercer);
    assertSame(geom, coercer.apply(Tuple.ofValues(onlyPointGeom, geom)));
  }

  @Test
  public void aStructWithAGeometryMemberIsEquivalentToGeometry() throws Exception {
    coercer = find(oneGeom, geometry);
    assertNotNull(coercer);
    assertSame(geom, coercer.apply(Tuple.ofValues(oneGeom, geom, "foo")));
  }

  @Test
  public void aStructWithAPointGeometryMemberIsEquivalentToGeometry() throws Exception {
    coercer = find(onePointGeom, geometry);
    assertNotNull(coercer);
    assertSame(geom, coercer.apply(Tuple.ofValues(onePointGeom, geom, "foo")));
  }

  @Test
  public void aStructWithAGeometryMemberIsEquivalentToGeometryEvenWhenChildMembersHaveGeom() {
    //
    coercer = find(oneGeomAndChildGeom, geometry);
    assertNotNull(coercer);
    assertSame(geom, coercer.apply(Tuple.ofValues(oneGeomAndChildGeom,
        Tuple.ofValues(oneGeom, gf.createPoint(new Coordinate(10, 10)), "foo"),
        geom)));
  }

  @Test
  public void aStructWithoutAGeometryMemberIsNotEquivalentToGeometryEvenIfChildDoesHaveGeom() {
    // coercer should not find geometries that are not in the inputStruct, even if they are children
    // of it
    coercer = find(noGeomButChildDoes, geometry);
    assertNull(coercer);

    coercer = find(noGeomButNullableChildDoes, geometry);
    assertNull(coercer);
  }

  @Test
  public void aStructWithNullableGeometryMemberIsEquivalentToNullableGeometry() throws Exception {
    coercer = find(nullableGeom, Nullable.GEOMETRY);
    assertNotNull(coercer);
    assertSame(geom, coercer.apply(Tuple.ofValues(nullableGeom, geom)));
    assertNull(coercer.apply(Tuple.ofValues(nullableGeom, new Object[] {null})));
  }

  @Test
  public void aStructWithASingleGeometryMemberIsEquivalentToNullableGeometry() throws Exception {
    coercer = find(onlyGeom, Nullable.GEOMETRY);
    assertNotNull(coercer);
    assertSame(geom, coercer.apply(Tuple.ofValues(onlyGeom, geom)));
  }

  @Test
  public void aStructWithManyGeomMembersIsNotEquivalent() throws Exception {
    assertNull(find(manyGeom, Types.GEOMETRY));
  }

  @Test
  public void aStructWithoutAGeometryMemberIsNotEquivalentToGeometry() throws Exception {
    assertNull(find(noGeom, geometry));
  }

  @Test
  public void aStructWithANullableGeometryMemberIsNotEquivalentToGeometry() throws Exception {
    assertNull(find(nullableGeom, geometry));
  }

  private Coercer find(Type source, Type target) {
    return typeSet.findEquivalenceCoercer(source, target).orElse(null);
  }

}
