/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.eqrule;

import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

import java.util.Optional;

import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.TypeRulesTest;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinSet;
import nz.org.riskscape.engine.types.varule.EqualityRule;
import nz.org.riskscape.engine.types.varule.Variance;

public class SingleValueStructRuleTest extends TypeRulesTest {

  @Override
  protected void addTypes() {
    registry.addEquivalenceRule(new SingleValueStructRule());
    registry.addEquivalenceRule(new NumberToFloatingRule());
    registry.addVarianceRule(new EqualityRule());
  }

  @Test
  public void simpleToStruct_EqualTypesAreEquivalent() throws Exception {
    Struct struct = Types.TEXT.asStruct();
    Coercer c = typeSet.findEquivalenceCoercer(Types.TEXT, struct).orElse(null);
    assertNotNull(c);
    assertEquals(struct, c.getTargetType());
    assertEquals(Tuple.ofValues(struct, "foo"), c.apply("foo"));
  }

  @Test
  public void simpleToStruct_UnEqualTypesAreEquivalent() throws Exception {
    Struct struct = Types.INTEGER.asStruct();
    Coercer c = typeSet.findEquivalenceCoercer(Types.TEXT, struct).orElse(null);
    assertNull(c);
  }

  @Test
  public void structToSimple_EqualTypesAreEquivalent() throws Exception {
    Struct struct = Types.TEXT.asStruct();
    Coercer c = typeSet.findEquivalenceCoercer(struct, Types.TEXT).orElse(null);
    assertNotNull(c);
    assertEquals(Types.TEXT, c.getTargetType());
    assertEquals("foo", c.apply(Tuple.ofValues(struct, "foo")));
  }

  @Test
  public void structToSimple_WithIntegerToFloatingConverstion() throws Exception {
    Struct struct = Types.INTEGER.asStruct();
    Coercer c = typeSet.findEquivalenceCoercer(struct, Types.FLOATING).orElse(null);
    assertNotNull(c);
    assertEquals(Types.FLOATING, c.getTargetType());
    assertEquals(10.0D, c.apply(Tuple.ofValues(struct, 10L)));
  }

  @Test
  public void structToSimple_UnEqualTypesAreNotEquivalent() throws Exception {
    Struct struct = Types.INTEGER.asStruct();
    Coercer c = typeSet.findEquivalenceCoercer(struct, Types.TEXT).orElse(null);
    assertNull(c);
  }

  @Test
  public void structToSimple_CoVariantTypesAreEquivalent() throws Exception {
    WithinSet set = new WithinSet(Types.TEXT, "foo", "bar");
    Struct struct = set.asStruct();
    when(mockVarianceRule.test(typeSet, set, Types.TEXT)).thenReturn(Variance.COVARIANT);

    Coercer c = typeSet.findEquivalenceCoercer(struct, Types.TEXT).orElse(null);
    assertNotNull(c);
    assertEquals(Types.TEXT, c.getTargetType());
    assertEquals("foo", c.apply(Tuple.ofValues(struct, "foo")));
  }

  @Test
  public void structToSimple_InVariantTypesAreNotEquivalent() throws Exception {
    WithinSet set = new WithinSet(Types.TEXT, "foo", "bar");
    Struct struct = Struct.of("set", set);
    when(mockVarianceRule.test(typeSet, set, Types.TEXT)).thenReturn(Variance.INVARIANT);

    Coercer c = typeSet.findEquivalenceCoercer(struct, Types.TEXT).orElse(null);
    assertNull(c);
  }

  @Test
  public void structToSimple_EquivalentTypesAreEquivalent() throws Exception {
    Struct struct = Types.TEXT.asStruct();
    when(mockEqRule.getCoercer(any(), eq(Types.TEXT), eq(Types.INTEGER)))
      .thenReturn(Optional.of(Coercer.build(Types.TEXT, Types.INTEGER, str -> Long.valueOf((String)str))));

    Coercer c = typeSet.findEquivalenceCoercer(struct, Types.INTEGER).orElse(null);
    assertNotNull(c);
    assertEquals(Types.INTEGER, c.getTargetType());
    assertEquals(1L, c.apply(Tuple.ofValues(struct, "1")));
  }

  @Test
  public void simpleToStruct_CoVariantTypesAreEquivalent() throws Exception {
    WithinSet set = new WithinSet(Types.TEXT, "foo", "bar");
    Struct struct = Types.TEXT.asStruct();
    when(mockVarianceRule.test(any(), eq(set), eq(Types.TEXT))).thenReturn(Variance.COVARIANT);

    Coercer c = typeSet.findEquivalenceCoercer(struct, Types.TEXT).orElse(null);
    assertNotNull(c);
    assertEquals(Types.TEXT, c.getTargetType());
    assertEquals(c.apply(Tuple.ofValues(struct, "foo")), "foo");
  }

  @Test
  public void simpleToStruct_InVariantTypesAreNotEquivalent() throws Exception {
    WithinSet set = new WithinSet(Types.TEXT, "foo", "bar");
    Struct struct = Struct.of("set", set);
    when(mockVarianceRule.test(typeSet, Types.TEXT, set)).thenReturn(Variance.INVARIANT);

    Coercer c = typeSet.findEquivalenceCoercer(Types.TEXT, struct).orElse(null);
    assertNull(c);
  }

  @Test
  public void simpleToStruct_EquivalentTypesAreEquivalent() throws Exception {
    Struct struct = Types.INTEGER.asStruct();
    when(mockEqRule.getCoercer(any(), eq(Types.INTEGER), eq(Types.TEXT)))
      .thenReturn(Optional.of(Coercer.build(Types.INTEGER, Types.TEXT, str -> str.toString())));

    Coercer c = typeSet.findEquivalenceCoercer(struct, Types.TEXT).orElse(null);
    assertNotNull(c);
    assertEquals(Types.TEXT, c.getTargetType());
    assertEquals("1", c.apply(Tuple.ofValues(struct, 1L)));
  }

}
