/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.eqrule;

import static org.hamcrest.Matchers.*;
import static org.hamcrest.Matchers.contains;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

import java.util.List;
import java.util.Optional;

import org.junit.Test;

import nz.org.riskscape.engine.Tuple;
import nz.org.riskscape.engine.problem.GeneralProblems;
import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.Struct;
import nz.org.riskscape.engine.types.TypeProblems;
import nz.org.riskscape.engine.types.TypeRulesTest;
import nz.org.riskscape.engine.types.Types;
import nz.org.riskscape.engine.types.WithinSet;
import nz.org.riskscape.engine.types.varule.EqualityRule;
import nz.org.riskscape.engine.types.varule.Variance;
import nz.org.riskscape.problem.Problem;

public class PartialStructRuleTest extends TypeRulesTest {

  @Override
  protected void addTypes() {
    registry.addVarianceRule(new EqualityRule());
    registry.addEquivalenceRule(new PartialStructRule());
  }

  @Test
  public void sameStructsGetAnIdentityCoercer() throws Exception {
    Struct struct = Struct.of("foo", Types.TEXT);
    Coercer matched = typeSet.findEquivalenceCoercer(struct, struct).orElse(null);
    assertNotNull(matched);
    // it's the identity coercer
    assertThat(matched.isIdentity(), is(true));
    // don't report this as a problem to the user though
    assertThat(getProblems(struct, struct), empty());
  }

  @Test
  public void structsWithSameMembersAreEquivalent() throws Exception {
    Struct source = Struct.of("foo", Types.TEXT, "bar", Types.INTEGER);
    Struct target = Struct.of("bar", Types.INTEGER, "foo", Types.TEXT);
    Coercer matched = typeSet.findEquivalenceCoercer(source, target).orElse(null);
    assertNotNull(matched);
    assertThat(matched.isIdentity(), is(false));  // actual coercing here
    Tuple expected = Tuple.ofValues(target, 1L, "foo");
    assertEquals(expected, matched.apply(Tuple.ofValues(source, "foo", 1L)));
    assertThat(getProblems(source, target), empty());
  }

  @Test
  public void structsWithDifferentMemberTypesAreNotEquivalent() throws Exception {
    Struct source = Struct.of("foo", Types.TEXT, "bar", Types.INTEGER);
    Struct target = Struct.of("foo", Types.INTEGER, "bar", Types.TEXT);
    Coercer matched = typeSet.findEquivalenceCoercer(source, target).orElse(null);
    assertNull(matched);
    assertThat(getProblems(source, target), contains(
        TypeProblems.get().mismatch(target.getEntry("foo"), Types.INTEGER, Types.TEXT),
        TypeProblems.get().mismatch(target.getEntry("bar"), Types.TEXT, Types.INTEGER)
    ));
  }

  @Test
  public void structsWithExactSameMembersAreEquivalent() throws Exception {
    Struct source = Struct.of("foo", Types.TEXT, "bar", Types.INTEGER);
    Struct target = Struct.of("foo", Types.TEXT, "bar", Types.INTEGER);

    Coercer matched = typeSet.findEquivalenceCoercer(source, target).orElse(null);
    assertNotNull(matched);
    Tuple given = Tuple.ofValues(source, "foo", 1L);
    Tuple expected = Tuple.ofValues(target, "foo", 1L);
    assertEquals(expected, matched.apply(given));
    assertSame(expected.getStruct(), target);
    assertThat(getProblems(source, target), empty());
  }

  @Test
  public void structsWithSuperfluousSourceMembersAreEquivalent() throws Exception {
    Struct source = Struct.of("foo", Types.TEXT, "bar", Types.INTEGER, "baz", Types.FLOATING);
    Struct target = Struct.of("bar", Types.INTEGER, "foo", Types.TEXT);
    Coercer matched = typeSet.findEquivalenceCoercer(source, target).orElse(null);
    assertNotNull(matched);
    Tuple expected = Tuple.ofValues(target, 1L, "foo");
    assertEquals(expected, matched.apply(Tuple.ofValues(source, "foo", 1L, 99D)));
    assertThat(getProblems(source, target), empty());
  }

  @Test
  public void structsWithSuperfluousTargetMembersAreNotEquivalent() throws Exception {
    Struct source = Struct.of("foo", Types.TEXT, "bar", Types.INTEGER);
    Struct target = Struct.of("bar", Types.INTEGER, "foo", Types.TEXT, "baz", Types.FLOATING);
    Coercer matched = typeSet.findEquivalenceCoercer(source, target).orElse(null);
    assertNull(matched);
    assertThat(getProblems(source, target), contains(
        GeneralProblems.get().required(target.getEntry("baz"))
    ));
  }

  @Test
  public void structsWithSuperfluousNullableTargetMembersAreEquivalent() throws Exception {
    Struct source = Struct.of("foo", Types.TEXT, "bar", Types.INTEGER);
    Struct target = Struct.of("bar", Types.INTEGER, "foo", Types.TEXT, "baz", Nullable.FLOATING);
    Coercer matched = typeSet.findEquivalenceCoercer(source, target).orElse(null);
    assertNotNull(matched);
    Tuple expected = Tuple.ofValues(target, 1L, "foo", null);
    assertEquals(expected, matched.apply(Tuple.ofValues(source, "foo", 1L)));
    assertThat(getProblems(source, target), empty());
  }

  @Test
  public void structsWithCovariantMembersAreEquivalent() throws Exception {
    WithinSet set = new WithinSet(Types.TEXT, "foo", "bar", "baz");
    Struct source = Struct.of("foo", set);
    Struct target = Struct.of("foo", Types.TEXT);

    when(mockVarianceRule.test(any(), eq(set), eq(Types.TEXT))).thenReturn(Variance.COVARIANT);

    Coercer matched = typeSet.findEquivalenceCoercer(source, target).orElse(null);
    assertNotNull(matched);
    Tuple expected = Tuple.ofValues(target, "foo");
    assertEquals(expected, matched.apply(Tuple.ofValues(source, "foo")));
    assertThat(getProblems(source, target), empty());
  }

  @Test
  public void structsWithEquivalentMembersAreEquivalent() throws Exception {
    WithinSet set = new WithinSet(Types.TEXT, "foo", "bar", "baz");
    Struct source = Struct.of("foo", set);
    Struct target = Struct.of("foo", Types.TEXT);

    when(mockEqRule.getCoercer(any(), eq(set), eq(Types.TEXT))).
      thenReturn(Optional.of(Coercer.build(set, Types.TEXT, p -> "baz!")));

    Coercer matched = typeSet.findEquivalenceCoercer(source, target).orElse(null);
    assertNotNull(matched);
    Tuple expected = Tuple.ofValues(target, "baz!");
    assertEquals(expected, matched.apply(Tuple.ofValues(source, "foo")));
    assertThat(getProblems(source, target), empty());
  }

  private List<Problem> getProblems(Struct source, Struct target) {
    return PartialStructRule.describeMismatch(typeSet, source, target);
  }
}
