/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types.eqrule;

import static org.hamcrest.Matchers.*;
import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

import java.util.Optional;

import org.junit.Test;
import org.mockito.Mockito;

import nz.org.riskscape.engine.types.Nullable;
import nz.org.riskscape.engine.types.RSList;
import nz.org.riskscape.engine.types.Type;
import nz.org.riskscape.engine.types.TypeRulesTest;
import nz.org.riskscape.engine.types.Types;


public class NullableRuleTest extends TypeRulesTest {

  Coercer mockCoercer = mock(Coercer.class);

  Type source = RSList.create(Types.TEXT);
  Type target = RSList.create(Types.INTEGER);

  @Override
  protected void addTypes() {
    registry.addEquivalenceRule(new NullableRule());


    when(mockEqRule.getCoercer(Mockito.any(), same(source), same(target)))
      .thenReturn(Optional.of(mockCoercer));

    when(mockCoercer.getTargetType()).thenReturn(target);
    when(mockCoercer.apply("foo")).thenReturn(99L);
  }

  @Test
  public void nullableRuleAppliesIfTheTargetTypeIsNullable() throws Exception {
    Coercer c = typeSet.findEquivalenceCoercer(source, Nullable.of(target)).orElse(null);
    assertEquals(99L, c.apply("foo"));
  }

  @Test
  public void nullableRuleAppliesIfTheTargetAndASourceTypeAreNullable() throws Exception {
    Coercer c = typeSet.findEquivalenceCoercer(Nullable.of(source), Nullable.of(target)).orElse(null);
    assertEquals(99L, c.apply("foo"));
  }

  @Test
  public void nullableRuleDoesNotApplyIfTheTargetTypeIsNotNullable() throws Exception {
    assertFalse(typeSet.findEquivalenceCoercer(Nullable.of(source), target).isPresent());
  }

  @Test
  public void nullableRuleDoesNotApplyIfContainedTypesAreNotEquivalent() throws Exception {
    Type otherSource = Types.GEOMETRY;
    assertFalse(typeSet.findEquivalenceCoercer(otherSource, Nullable.of(target)).isPresent());
    verify(mockEqRule).getCoercer(any(), same(otherSource), same(target));
  }

  @Test
  public void nullRuleReturnsNullableChildCoercerTargetType() {
    // contrived example. the target type from the nullable coercer should be a nullable version
    // of what the child coercer returns.
    // in practice this is to ensure that wrapping type information is not lost needlessly. For instance
    // referenced geometries, or specific geometry types where a Types.GEOMETRY is required.
    when(mockCoercer.getTargetType()).thenReturn(Types.GEOMETRY);

    Coercer c = typeSet.findEquivalenceCoercer(source, Nullable.of(target)).orElse(null);
    assertThat(c.getTargetType(), is(Nullable.GEOMETRY));

    c = typeSet.findEquivalenceCoercer(Nullable.of(source), Nullable.of(target)).orElse(null);
    assertThat(c.getTargetType(), is(Nullable.GEOMETRY));
  }

}
