/*
 * RiskScape™ Copyright New Zealand Institute for Earth Science Limited
 * (Earth Sciences New Zealand) is distributed for research purposes only
 * under the terms of AGPLv3.
 *
 * RiskScape™ Copyright 2025 New Zealand Institute for Earth Science
 * Limited (Earth Sciences New Zealand). All rights reserved. Source code
 * available under the AGPLv3.
 * 
 * This program is free software: you can redistribute it and/or modify it under
 *  the terms of the GNU Affero General Public License as published by the Free
 *  Software Foundation, either version 3 of the License, or (at your option) any
 *  later version.
 * 
 * This program is distributed for RESEARCH PURPOSES ONLY, in the hope that it will
 * be useful for research and education initiatives.
 * 
 * If you are not a researcher, or you are a researcher who wishes to use this
 * program on terms other than AGPLv3 (including those who wish to restrict the
 * distribution of any source code created using this program), please contact:
 * https://riskscape.org.nz
 * 
 * This program is distributed WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Affero General Public License for more details.  You should have received a copy
 * of the GNU Affero General Public License along with this program.  If not, see
 * <http://www.gnu.org/licenses/>.
 * 
 * By way of summary only, under the AGPLv3:
 *     • Permissions of this strongest copyleft license are conditioned
 *       on making available complete source code of licensed works and
 *       modifications, which include larger works using a licensed work,
 *       under the same license.
 *     • Copyright and license notices must be preserved.
 *     • Contributors provide an express grant of patent rights.
 *     • When a modified version is used to provide a service over a
 *       network, the complete source code of the modified version must be made
 *       available.
 */
package nz.org.riskscape.engine.types;

import static org.junit.Assert.*;
import static org.mockito.ArgumentMatchers.*;
import static org.mockito.Mockito.*;

import org.junit.Before;
import org.junit.Test;

public class WithinRangeTest {

  Rangeable rangeable;
  Comparable<?> min;
  Comparable<?> max;
  WithinRange type;

  @Before
  public void setup() {
    rangeable = mock(Rangeable.class);
    min = 12.4F;
    max = 4.6F;

    when(rangeable.coerce(any(Float.class))).thenAnswer(a -> {
      return Float.valueOf(a.getArgument(0).toString());
    });

    doReturn(Float.class).when(rangeable).internalType();

    type = new WithinRange(rangeable, min, max);
  }

  @Test
  public void acceptsAnyRangeableTypeWithAMinAndMax() {
    assertEquals(min, type.getMin());
    assertEquals(max, type.getMax());
  }

  // FIXME this should pass, but fixing it causes a bunch of tests all over the place to fail due to faulty
  // or accidentally working type munging that predates rl
//  @Test
//  public void asStructWrapsTypeInAStruct() throws Exception {
//    Struct struct = type.asStruct();
//    assertSame(type, struct.getMember("value").get().type);
//  }

  @Test(expected=IllegalArgumentException.class)
  public void raisesAnExceptionIfUnderlyingTypeNotComparable() {
    Rangeable otherRangeable = mock(Rangeable.class);

    doReturn(Object.class).when(otherRangeable).internalType();

    new WithinRange(otherRangeable, min, max);
  }

  @Test(expected=CoercionException.class)
  public void doesNotAcceptNull() {
    type.coerce(null);
  }

  @Test(expected=CoercionException.class)
  public void throwsCoercionExceptionIfLessThanMin() {
    type.coerce(Float.valueOf(4));
  }

  @Test(expected=CoercionException.class)
  public void throwsCoercisonExceptionIfGreaterThanMax() {
    type.coerce(Float.valueOf(13));
  }

  @Test(expected=CoercionException.class)
  public void throwsCoercionExceptionFromUnderlyingType() {
    String badValue = "naughty!";

    doThrow(new CoercionException(badValue, rangeable)).when(rangeable).coerce(same(badValue));

    type.coerce(badValue);
  }

}
